local addonName = ...

-------------------------------------------------
-- Cesta k fontu
-------------------------------------------------
local CS_FONT_PATH = "Interface\\AddOns\\MaterialBank\\Fonts\\FRIZQT.ttf"

-- ------------------------------------------------
-- Barva “linek”
-- ------------------------------------------------
local MBM_LINE_R, MBM_LINE_G, MBM_LINE_B, MBM_LINE_A = 0.42, 0.28, 0.14, 0.95

-- ------------------------------------------------
-- Barva “kusú”
-- ------------------------------------------------
local MBM_COUNT_R, MBM_COUNT_G, MBM_COUNT_B, MBM_COUNT_A = 1.0, 0.82, 0.25, 1.0

-- ------------------------------------------------
-- Barvy pro MONEY řádek
-- ------------------------------------------------
local MBM_MONEY_LABEL_R, MBM_MONEY_LABEL_G, MBM_MONEY_LABEL_B = 0.57, 0.83, 0.36

-- ------------------------------------------------
-- Barvy pro g/s/c
-- ------------------------------------------------
local MBM_MONEY_GOLD_R,   MBM_MONEY_GOLD_G,   MBM_MONEY_GOLD_B   = 1.00, 0.82, 0.00 -- žlutá
local MBM_MONEY_SILVER_R, MBM_MONEY_SILVER_G, MBM_MONEY_SILVER_B = 0.75, 0.75, 0.75 -- šedá
local MBM_MONEY_COPPER_R, MBM_MONEY_COPPER_G, MBM_MONEY_COPPER_B = 0.95, 0.55, 0.20 -- oranžová

local function MBM_RGBToHex(r, g, b)
    r = math.floor((tonumber(r) or 1) * 255 + 0.5)
    g = math.floor((tonumber(g) or 1) * 255 + 0.5)
    b = math.floor((tonumber(b) or 1) * 255 + 0.5)
    if r < 0 then r = 0 elseif r > 255 then r = 255 end
    if g < 0 then g = 0 elseif g > 255 then g = 255 end
    if b < 0 then b = 0 elseif b > 255 then b = 255 end
    return string.format("%02x%02x%02x", r, g, b)
end

local function MBM_ColorText(text, r, g, b)
    return "|cff" .. MBM_RGBToHex(r, g, b) .. (text or "") .. "|r"
end

-- tisícové oddělovače pro GOLD
local function MBM_FormatGoldWithCommas(n)
    n = tonumber(n) or 0
    if n < 0 then n = 0 end

    local s = tostring(math.floor(n))
    local out = {}

    local len = #s
    local first = len % 3
    if first == 0 then first = 3 end

    table.insert(out, string.sub(s, 1, first))

    local i = first + 1
    while i <= len do
        table.insert(out, string.sub(s, i, i + 2))
        i = i + 3
    end

    return table.concat(out, ",")
end

local function MBM_FormatMoneyColored(copper)
    copper = tonumber(copper) or 0
    if copper < 0 then copper = 0 end

    local g = math.floor(copper / 10000)
    local s = math.floor((copper % 10000) / 100)
    local c = copper % 100

    return
        MBM_FormatGoldWithCommas(g) .. MBM_ColorText("g", MBM_MONEY_GOLD_R,   MBM_MONEY_GOLD_G,   MBM_MONEY_GOLD_B) .. " " ..
        tostring(s)                 .. MBM_ColorText("s", MBM_MONEY_SILVER_R, MBM_MONEY_SILVER_G, MBM_MONEY_SILVER_B) .. " " ..
        tostring(c)                 .. MBM_ColorText("c", MBM_MONEY_COPPER_R, MBM_MONEY_COPPER_G, MBM_MONEY_COPPER_B)
end

local function MBM_SetLineColor(tex)
    if not tex then
        return
    end
    tex:SetTexture(MBM_LINE_R, MBM_LINE_G, MBM_LINE_B, MBM_LINE_A)
end

local function MBM_GetLang()
    if MaterialBankCache and type(MaterialBankCache._lang) == "string" then
        if MaterialBankCache._lang == "cs" or MaterialBankCache._lang == "en" then
            return MaterialBankCache._lang
        end
    end

    return "en"
end

local function L(cs, en)
    if MBM_GetLang() == "en" then
        return en
    else
        return cs
    end
end

local function SetFsFont(fs, size, flags)
    if not fs then
        return
    end
    fs:SetFont(CS_FONT_PATH, size, flags or "")
end

local function MBM_CreateCoinIcon(parent, anchorTo, texPath, size, x, y)
    local t = parent:CreateTexture(nil, "OVERLAY")
    t:SetTexture(texPath)
    t:SetSize(size or 14, size or 14)
    t:SetPoint("LEFT", anchorTo, "RIGHT", x or 2, y or 0)
    t:Hide()
    return t
end

-------------------------------------------------
-- Kategorie – zrcadlo tabulky material_bank_category
-------------------------------------------------
local MBM_Categories = {
    [1] = {parent = 0, cs = "Obchodní zboží", en = "Trade Goods"},
    [2] = {parent = 1, cs = "Trade Goods", en = "Trade Goods"},
    [3] = {parent = 1, cs = "Parts", en = "Parts"},
    [4] = {parent = 1, cs = "Explosives", en = "Explosives"},
    [5] = {parent = 1, cs = "Devices", en = "Devices"},
    [6] = {parent = 1, cs = "Jewelcrafting", en = "Jewelcrafting"},
    [7] = {parent = 1, cs = "Cloth", en = "Cloth"},
    [8] = {parent = 1, cs = "Leather", en = "Leather"},
    [9] = {parent = 1, cs = "Metal & Stone", en = "Metal & Stone"},
    [10] = {parent = 1, cs = "Meat", en = "Meat"},
    [11] = {parent = 1, cs = "Herb", en = "Herb"},
    [12] = {parent = 1, cs = "Elemental", en = "Elemental"},
    [13] = {parent = 1, cs = "Other", en = "Other"},
    [14] = {parent = 1, cs = "Enchanting", en = "Enchanting"},
    [15] = {parent = 1, cs = "Materials", en = "Materials"},
    [16] = {parent = 1, cs = "Armor Enchantment", en = "Armor Enchantment"},
    [17] = {parent = 1, cs = "Weapon Enchantment", en = "Weapon Enchantment"},
    [18] = {parent = 0, cs = "Zbroj", en = "Armor"},
    [19] = {parent = 18, cs = "Miscellaneous", en = "Miscellaneous"},
    [20] = {parent = 18, cs = "Cloth", en = "Cloth"},
    [21] = {parent = 18, cs = "Leather", en = "Leather"},
    [22] = {parent = 18, cs = "Mail", en = "Mail"},
    [23] = {parent = 18, cs = "Plate", en = "Plate"},
    [24] = {parent = 18, cs = "Shield", en = "Shield"},
    [25] = {parent = 18, cs = "Libram", en = "Libram"},
    [26] = {parent = 18, cs = "Idol", en = "Idol"},
    [27] = {parent = 18, cs = "Totem", en = "Totem"},
    [28] = {parent = 18, cs = "Sigil", en = "Sigil"},
    [29] = {parent = 0, cs = "Zbraně", en = "Weapon"},
    [30] = {parent = 29, cs = "One-Handed Axes", en = "One-Handed Axes"},
    [31] = {parent = 29, cs = "Two-Handed Axes", en = "Two-Handed Axes"},
    [32] = {parent = 29, cs = "One-Handed Maces", en = "One-Handed Maces"},
    [33] = {parent = 29, cs = "Two-Handed Mace", en = "Two-Handed Mace"},
    [34] = {parent = 29, cs = "One-Handed Swords", en = "One-Handed Swords"},
    [35] = {parent = 29, cs = "Two-Handed Sword", en = "Two-Handed Sword"},
    [36] = {parent = 29, cs = "Polearm", en = "Polearm"},
    [37] = {parent = 29, cs = "Staff", en = "Staff"},
    [38] = {parent = 29, cs = "Fist Weapon", en = "Fist Weapon"},
    [39] = {parent = 29, cs = "Dagger", en = "Dagger"},
    [40] = {parent = 29, cs = "Bow", en = "Bow"},
    [41] = {parent = 29, cs = "Gun", en = "Gun"},
    [42] = {parent = 29, cs = "Crossbow", en = "Crossbow"},
    [43] = {parent = 29, cs = "Thrown", en = "Thrown"},
    [44] = {parent = 29, cs = "Wand", en = "Wand"},
    [45] = {parent = 29, cs = "Miscellaneous", en = "Miscellaneous"},
    [46] = {parent = 29, cs = "Fishing Pole", en = "Fishing Pole"},
    [47] = {parent = 0, cs = "Spotřební zboží", en = "Consumable"},
    [48] = {parent = 47, cs = "Food & Drink", en = "Food & Drink"},
    [49] = {parent = 47, cs = "Potion", en = "Potion"},
    [50] = {parent = 47, cs = "Elixir", en = "Elixir"},
    [51] = {parent = 47, cs = "Flask", en = "Flask"},
    [52] = {parent = 47, cs = "Item Enhancement", en = "Item Enhancement"},
    [53] = {parent = 47, cs = "Bandage", en = "Bandage"},
    [54] = {parent = 47, cs = "Scroll", en = "Scroll"},
    [55] = {parent = 47, cs = "Other", en = "Other"},
    [56] = {parent = 47, cs = "Consumable", en = "Consumable"},
    [57] = {parent = 0, cs = "Gemy", en = "Gem"},
    [58] = {parent = 57, cs = "Red", en = "Red"},
    [59] = {parent = 57, cs = "Blue", en = "Blue"},
    [60] = {parent = 57, cs = "Yellow", en = "Yellow"},
    [61] = {parent = 57, cs = "Purple", en = "Purple"},
    [62] = {parent = 57, cs = "Green", en = "Green"},
    [63] = {parent = 57, cs = "Orange", en = "Orange"},
    [64] = {parent = 57, cs = "Meta", en = "Meta"},
    [65] = {parent = 57, cs = "Simple", en = "Simple"},
    [66] = {parent = 57, cs = "Prismatic", en = "Prismatic"},
    [67] = {parent = 0, cs = "Recepty", en = "Recipe"},
    [68] = {parent = 67, cs = "Leatherworking", en = "Leatherworking"},
    [69] = {parent = 67, cs = "Tailoring", en = "Tailoring"},
    [70] = {parent = 67, cs = "Engineering", en = "Engineering"},
    [71] = {parent = 67, cs = "Blacksmithing", en = "Blacksmithing"},
    [72] = {parent = 67, cs = "Alchemy", en = "Alchemy"},
    [73] = {parent = 67, cs = "Enchanting", en = "Enchanting"},
    [74] = {parent = 67, cs = "Jewelcrafting", en = "Jewelcrafting"},
    [75] = {parent = 67, cs = "Inscription", en = "Inscription"},
    [76] = {parent = 67, cs = "Cooking", en = "Cooking"},
    [77] = {parent = 67, cs = "First Aid", en = "First Aid"},
    [78] = {parent = 67, cs = "Fishing", en = "Fishing"},
    [79] = {parent = 67, cs = "Book", en = "Book"},
    [80] = {parent = 0, cs = "Glyfy", en = "Glyph"},
    [81] = {parent = 80, cs = "Warrior", en = "Warrior"},
    [82] = {parent = 80, cs = "Paladin", en = "Paladin"},
    [83] = {parent = 80, cs = "Hunter", en = "Hunter"},
    [84] = {parent = 80, cs = "Rogue", en = "Rogue"},
    [85] = {parent = 80, cs = "Priest", en = "Priest"},
    [86] = {parent = 80, cs = "Death Knight", en = "Death Knight"},
    [87] = {parent = 80, cs = "Shaman", en = "Shaman"},
    [88] = {parent = 80, cs = "Mage", en = "Mage"},
    [89] = {parent = 80, cs = "Warlock", en = "Warlock"},
    [90] = {parent = 80, cs = "Druid", en = "Druid"},
    [91] = {parent = 0, cs = "Brašny", en = "Container"},
    [92] = {parent = 91, cs = "Bag", en = "Bag"},
    [93] = {parent = 91, cs = "Soul Bag", en = "Soul Bag"},
    [94] = {parent = 91, cs = "Herb Bag", en = "Herb Bag"},
    [95] = {parent = 91, cs = "Enchanting Bag", en = "Enchanting Bag"},
    [96] = {parent = 91, cs = "Engineering Bag", en = "Engineering Bag"},
    [97] = {parent = 91, cs = "Gem Bag", en = "Gem Bag"},
    [98] = {parent = 91, cs = "Mining Bag", en = "Mining Bag"},
    [99] = {parent = 91, cs = "Leatherworking Bag", en = "Leatherworking Bag"},
    [100] = {parent = 91, cs = "Inscription Bag", en = "Inscription Bag"},
    [101] = {parent = 0, cs = "Ostatní", en = "Miscellaneous"},
    [102] = {parent = 101, cs = "Junk", en = "Junk"},
    [103] = {parent = 101, cs = "Reagent", en = "Reagent"},
    [104] = {parent = 101, cs = "Pet", en = "Pet"},
    [105] = {parent = 101, cs = "Holiday", en = "Holiday"},
    [106] = {parent = 101, cs = "Other", en = "Other"},
    [107] = {parent = 101, cs = "Mount", en = "Mount"},
	[108] = {parent = 101, cs = "Quest", en = "Quest"}
}

local function MBM_GetCategoryName(catId)
    local info = MBM_Categories[catId]
    if not info then
        local cs = "Kategorie " .. tostring(catId)
        local en = "Category " .. tostring(catId)
        return cs, en
    end

    local cs = info.cs or ("Kategorie " .. tostring(catId))
    local en = info.en or ("Category " .. tostring(catId))
    return cs, en
end

local function MBM_FindRootCategory(catId)
    local current = catId
    local safety = 0
    while current and current ~= 0 and safety < 10 do
        safety = safety + 1
        local info = MBM_Categories[current]
        if not info then
            break
        end
        if not info.parent or info.parent == 0 then
            return current
        end
        current = info.parent
    end
    return nil
end

-------------------------------------------------
-- Cache pro aktuální postavu
-------------------------------------------------
local function MBM_GetCharKey()
    local name = UnitName("player")
    local realm = GetRealmName() or "UnknownRealm"
    return name .. "-" .. realm
end

local function MBM_GetCharCache()
    MaterialBankCache = MaterialBankCache or {}
    local key = MBM_GetCharKey()
    MaterialBankCache[key] = MaterialBankCache[key] or {}
    return MaterialBankCache[key]
end

-------------------------------------------------
-- Výjimky pro vklad (Pouze zda je vidět!!!)
-------------------------------------------------
local MBM_AllowQuestItems = {
}

local MBM_AllowSoulboundItems = {
    [44874] = true,
    [44877] = true,
    [12753] = true,
    [44050] = true,
    [44137] = true,
    [44131] = true,
    [44130] = true,
    [44132] = true,
    [44129] = true,
    [44141] = true,
    [44140] = true,
    [44138] = true,
    [44139] = true
}

local MBM_AllowQuestSoulboundItems = {
}

-------------------------------------------------
-- Blacklist pro vklad (NIKDY nezobrazovat ve vkladu) - podle itemId
-------------------------------------------------
local MBM_BlockDepositItems = {
    [17030] = true -- Ankh
}

-------------------------------------------------
-- Forward deklarace
-------------------------------------------------
local MBM_MainFrame
local MBM_ModeButtons = {}
local MBM_DepositMoneyGoldBox, MBM_DepositMoneySilverBox, MBM_DepositMoneyCopperBox
local MBM_DepositMoneyButton
local MBM_WithdrawMoneyGoldBox, MBM_WithdrawMoneySilverBox, MBM_WithdrawMoneyCopperBox
local MBM_WithdrawMoneyButton
local MBM_StoredMoneyText
local MBM_DepositMoneyGoldLabel, MBM_DepositMoneySilverLabel, MBM_DepositMoneyCopperLabel
local MBM_WithdrawMoneyGoldLabel, MBM_WithdrawMoneySilverLabel, MBM_WithdrawMoneyCopperLabel
local MBM_BuildCategoryList
local MBM_RebuildItemList
local MBM_UpdateTopStripDepositControls
local MBM_SearchBox
local MBM_SearchHint
local MBM_SearchClearBtn
local MBM_DepositAllBtn
local MBM_DepositAllFrame
local MBM_TitleFrame
local MBM_TitleText
local MBM_UpdateTitleBar
local MBM_CATBTN_CFG
local MBM_SkinCategoryButton
local MBM_SetCategoryButtonActive

local function MBM_ApplyLocaleStaticTexts()
	if MBM_ModeButtons and MBM_ModeButtons.withdraw then
		local b = MBM_ModeButtons.withdraw
		if b._mbmLabel then
			b._mbmLabel:SetText(L("Výběr", "Withdraw"))
		else
			b:SetText(L("Výběr", "Withdraw"))
		end
	end
	if MBM_ModeButtons and MBM_ModeButtons.deposit then
		local b = MBM_ModeButtons.deposit
		if b._mbmLabel then
			b._mbmLabel:SetText(L("Vklad", "Deposit"))
		else
			b:SetText(L("Vklad", "Deposit"))
		end
	end

    -- Zobrazit skryté:
    if MBM_MainFrame and MBM_MainFrame.showHiddenLabel then
        MBM_MainFrame.showHiddenLabel:SetText(L("Zobrazit skryté:", "Show hidden:"))
    end

    -- Nápověda ve vyhledávání (placeholder)
    if MBM_SearchHint then
        MBM_SearchHint:SetText(L("Hledat...", "Search..."))
    end

	-- Uložit vše tlačítko
	if MBM_DepositAllBtn then
		local t
		if MBM_DepositAllFrame and MBM_DepositAllFrame:IsShown() then
			t = L("Ukládám...", "Depositing...")
		else
			t = L("Uložit vše", "Deposit all")
		end
	
		if MBM_DepositAllBtn._mbmLabel then
			MBM_DepositAllBtn._mbmLabel:SetText(t)
		else
			MBM_DepositAllBtn:SetText(t)
		end
	end

    -- Titulek
    if MBM_UpdateTitleBar then
        MBM_UpdateTitleBar()
    end
end

-------------------------------------------------
-- Tooltip sken
-------------------------------------------------
local MBM_ScanTooltip = CreateFrame("GameTooltip", "MBM_ScanTooltip", UIParent, "GameTooltipTemplate")
MBM_ScanTooltip:SetOwner(UIParent, "ANCHOR_NONE")

local function MBM_GetBagItemFlags(bag, slot)
    local tt = MBM_ScanTooltip
    tt:SetOwner(UIParent, "ANCHOR_NONE")
    tt:ClearLines()
    tt:SetBagItem(bag, slot)

    local isSoulbound = false
    local isQuest = false
    local isConjured = false

    for i = 1, tt:NumLines() do
        local left = _G["MBM_ScanTooltipTextLeft" .. i]
        if left then
            local text = left:GetText()
            if text and text ~= "" then
                if text == ITEM_SOULBOUND then
                    isSoulbound = true
                end
                if text == ITEM_BIND_QUEST then
                    isQuest = true
                end
                if text == ITEM_CONJURED then
                    isConjured = true
                end
            end
        end
    end

    return isSoulbound, isQuest, isConjured
end

local function MBM_ShouldIncludeItemForDeposit(bag, slot, itemId, itemLink)
    if not itemId or not itemLink then
        return false
    end

    if MBM_BlockDepositItems[itemId] == true then
        return false
    end

    local isSoulbound, isQuest, isConjured = MBM_GetBagItemFlags(bag, slot)

    if isConjured then
        return false
    end

    if isQuest and isSoulbound then
        return MBM_AllowQuestSoulboundItems[itemId] == true
    end

    if isQuest then
        return MBM_AllowQuestItems[itemId] == true
    end

    if isSoulbound then
        return MBM_AllowSoulboundItems[itemId] == true
    end

    return true
end

local function MBM_BuildBagItemList()
    local counts = {}

    local maxBags = NUM_BAG_SLOTS or 4
    for bag = 0, maxBags do
        local slots = GetContainerNumSlots(bag)
        if slots and slots > 0 then
            for slot = 1, slots do
                local _, itemCount, _, _, _, _, itemLink = GetContainerItemInfo(bag, slot)
                if itemLink and itemCount and itemCount > 0 then
                    local itemId = tonumber(string.match(itemLink, "item:(%d+)"))
                    if itemId then
                        if MBM_ShouldIncludeItemForDeposit(bag, slot, itemId, itemLink) then
                            counts[itemId] = (counts[itemId] or 0) + itemCount
                        end
                    end
                end
            end
        end
    end

    local items = {}
    for itemId, total in pairs(counts) do
        table.insert(items, {id = itemId, total = total})
    end
    table.sort(
        items,
        function(a, b)
            return a.id < b.id
        end
    )
    return items
end

-------------------------------------------------
-- SavedVariables: minimapa
-------------------------------------------------
local function MBM_GetMinimapSV()
    MaterialBankCache = MaterialBankCache or {}
    MaterialBankCache._minimap = MaterialBankCache._minimap or {}
    local sv = MaterialBankCache._minimap
    if not sv.angle then
        sv.angle = 210
    end
    return sv
end

local function MBM_GetModeSV()
    MaterialBankCache = MaterialBankCache or {}
    MaterialBankCache._mode = MaterialBankCache._mode or "withdraw"
    return MaterialBankCache._mode
end

local function MBM_SetModeSV(mode)
    MaterialBankCache = MaterialBankCache or {}
    MaterialBankCache._mode = mode
end

-------------------------------------------------
-- SavedVariables: UI (skryté itemy jen pro deposit)
-------------------------------------------------
local function MBM_GetUiSV()
    MaterialBankCache = MaterialBankCache or {}
    MaterialBankCache._ui = MaterialBankCache._ui or {}

    local key = MBM_GetCharKey()
    MaterialBankCache._ui[key] = MaterialBankCache._ui[key] or {}

    local ui = MaterialBankCache._ui[key]
    ui.depositHidden = ui.depositHidden or {}
    if ui.showHiddenDeposit == nil then
        ui.showHiddenDeposit = false
    end

    return ui
end

local function MBM_IsDepositHidden(itemId)
    local ui = MBM_GetUiSV()
    return ui.depositHidden[itemId] == true
end

local function MBM_SetDepositHidden(itemId, hidden)
    local ui = MBM_GetUiSV()
    if hidden then
        ui.depositHidden[itemId] = true
    else
        ui.depositHidden[itemId] = nil
    end
end

local function MBM_GetShowHiddenDeposit()
    local ui = MBM_GetUiSV()
    return ui.showHiddenDeposit == true
end

local function MBM_SetShowHiddenDeposit(v)
    local ui = MBM_GetUiSV()
    ui.showHiddenDeposit = v and true or false
end

-------------------------------------------------
-- SavedVariables: peníze (per account + frakce)
-------------------------------------------------
local function MBM_GetMoneyKey()
    local faction = UnitFactionGroup("player") or "Neutral"
    return faction
end

local function MBM_GetMoneySV()
    MaterialBankCache = MaterialBankCache or {}
    MaterialBankCache._money = MaterialBankCache._money or {}

    local key = MBM_GetMoneyKey()
    MaterialBankCache._money[key] = MaterialBankCache._money[key] or { copper = 0 }

    return MaterialBankCache._money[key]
end

local function MBM_GetStoredMoneyCopper()
    local sv = MBM_GetMoneySV()
    return sv.copper or 0
end

local function MBM_FormatMoney(copper)
    copper = tonumber(copper) or 0
    if copper < 0 then copper = 0 end

    local g = math.floor(copper / 10000)
    local s = math.floor((copper % 10000) / 100)
    local c = copper % 100

    return string.format("%dg %ds %dc", g, s, c)
end

function MBM_SetStoredMoneyFromServer(copper)
    local sv = MBM_GetMoneySV()
    sv.copper = math.max(0, tonumber(copper) or 0)

    if MBM_MainFrame and MBM_MainFrame:IsShown() then
        MBM_UpdateTopStripDepositControls()
    end
end

-------------------------------------------------
-- Lokální proměnné
-------------------------------------------------
local MBM_MinimapButton
local MBM_SelectedMode = "withdraw"
local MBM_OpenFromMinimap = false
local MBM_ChatFilterInstalled = false

local MBM_CategoryButtons = {}
local MBM_CategoryState = {}
local MBM_SelectedCategory

local MBM_ItemScrollFrame
local MBM_ItemContent
local MBM_ItemRows = {}
local MBM_ItemInfoPending = {}

local MBM_CategoryScrollFrame
local MBM_CategoryContent

local MBM_AmountRemember = {}

local MBM_UniqueEditBoxId = 0
local MBM_StartDepositAll

local MBM_RequestHardReload

local MBM_ActiveEditBox
local MBM_WorldHooked = false

-------------------------------------------------
-- Debounced refresh pro Deposit list (bag updates)
-------------------------------------------------
local MBM_BagRefreshFrame = CreateFrame("Frame")
MBM_BagRefreshFrame:Hide()
MBM_BagRefreshFrame._t = 0
MBM_BagRefreshFrame._delay = 0.10

local function MBM_RequestDepositRefresh()
    if not MBM_MainFrame or not MBM_MainFrame:IsShown() then
        return
    end

    local mode = MBM_SelectedMode or MBM_GetModeSV() or "withdraw"
    if mode ~= "deposit" then
        return
    end

    if MBM_SelectedCategory ~= 0 then
        return
    end

    MBM_BagRefreshFrame._t = 0
    MBM_BagRefreshFrame:Show()
end

MBM_BagRefreshFrame:SetScript("OnUpdate", function(self, elapsed)
    self._t = (self._t or 0) + (elapsed or 0)
    if self._t < (self._delay or 0.10) then
        return
    end
    self:Hide()

    if MBM_MainFrame and MBM_MainFrame:IsShown() then
        MBM_RebuildItemList()
        if MBM_ItemScrollFrame then
            MBM_UpdateScrollState(MBM_ItemScrollFrame)
        end
    end
end)

local function MBM_ClearAnyEditFocus()
    local f = GetCurrentKeyBoardFocus and GetCurrentKeyBoardFocus() or nil
    if f and f.ClearFocus then
        f:ClearFocus()
    end

    if MBM_ActiveEditBox and MBM_ActiveEditBox.ClearFocus then
        MBM_ActiveEditBox:ClearFocus()
    end

    MBM_ActiveEditBox = nil
end

local function MBM_ApplyScrollBarSkin(scrollFrame)
    if not scrollFrame or not scrollFrame.GetName then
        return
    end
    local sb = _G[scrollFrame:GetName() .. "ScrollBar"]
    if not sb or sb._mbmSkinned then
        return
    end
    sb._mbmSkinned = true

    -------------------------------------------------
    -- Scrollbar
    -------------------------------------------------
    local up = sb.ScrollUpButton
    local down = sb.ScrollDownButton
    if up then
        up:SetNormalTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Up")
        up:SetPushedTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Down")
        up:SetDisabledTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Disabled")
        up:SetHighlightTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Highlight")
    end
    if down then
        down:SetNormalTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Up")
        down:SetPushedTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Down")
        down:SetDisabledTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Disabled")
        down:SetHighlightTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Highlight")
    end

    local thumb = sb.GetThumbTexture and sb:GetThumbTexture() or nil
    if thumb then
        thumb:SetTexture("Interface\\Buttons\\UI-ScrollBar-Knob")
        thumb:SetTexCoord(0.25, 0.75, 0.25, 0.75)
        thumb:SetSize(16, 24)
    end

    if sb.SetBackdrop then
        sb:SetBackdrop(
            {
                bgFile = "Interface\\Buttons\\UI-ScrollBar-Background",
                edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
                tile = true,
                tileSize = 16,
                edgeSize = 12,
                insets = {left = 3, right = 3, top = 3, bottom = 3}
            }
        )
        sb:SetBackdropColor(0, 0, 0, 0.35)
        sb:SetBackdropBorderColor(1, 1, 1, 0.25)
    end

    if not sb._mbmLeftWhiteLine then
        local line = scrollFrame:CreateTexture(nil, "OVERLAY")
        MBM_SetLineColor(line)
        line:SetWidth(2)

        line:SetPoint("TOPLEFT", sb, "TOPLEFT", -7, 21)
        line:SetPoint("BOTTOMLEFT", sb, "BOTTOMLEFT", -7, -29)

        sb._mbmLeftWhiteLine = line
    end

    -------------------------------------------------
    -- Border rám kolem scrollu
    -------------------------------------------------
    local border = CreateFrame("Frame", nil, scrollFrame)
    border:SetPoint("TOPLEFT", sb, "TOPLEFT", -4, 4)
    border:SetPoint("BOTTOMRIGHT", sb, "BOTTOMRIGHT", 4, -4)
    border:SetFrameLevel(sb:GetFrameLevel() - 1)
    border:SetBackdrop(
        {
            edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
            tile = true,
            tileSize = 16,
            edgeSize = 12,
            insets = {left = 3, right = 3, top = 3, bottom = 3}
        }
    )
    border:SetBackdropBorderColor(1, 1, 1, 0.35)
end

local function MBM_UpdateScrollState(scrollFrame)
    if not scrollFrame then
        return
    end

    if scrollFrame.UpdateScrollChildRect then
        scrollFrame:UpdateScrollChildRect()
    end

    local range = scrollFrame.GetVerticalScrollRange and (scrollFrame:GetVerticalScrollRange() or 0) or 0
    local cur = scrollFrame.GetVerticalScroll and (scrollFrame:GetVerticalScroll() or 0) or 0

    if cur > range and scrollFrame.SetVerticalScroll then
        scrollFrame:SetVerticalScroll(range)
        cur = range
    end

    local sb = _G[scrollFrame:GetName() .. "ScrollBar"]
    if not sb then
        return
    end

    sb:SetMinMaxValues(0, range)
    sb:SetValue(cur)

    local up = sb.ScrollUpButton
    local down = sb.ScrollDownButton

    if range <= 0 then
        if up then
            up:Disable()
        end
        if down then
            down:Disable()
        end
        return
    end

    if up then
        if cur <= 0 then
            up:Disable()
        else
            up:Enable()
        end
    end
    if down then
        if cur >= range then
            down:Disable()
        else
            down:Enable()
        end
    end
end

-------------------------------------------------
-- Kategorie z reálných dat
-------------------------------------------------
local function MBM_GetCategories()
    local mode = MBM_SelectedMode or MBM_GetModeSV() or "withdraw"

    if mode == "deposit" then
        return {
            {
                id = 0,
                name_cs = "Inventář",
                name_en = "Inventory",
                subs = {}
            }
        }
    end

    local charCache = MBM_GetCharCache()
    local byRoot = {}
    local hasUncategorized = false

    for itemId, data in pairs(charCache) do
        if type(data) == "table" and data.total and data.total > 0 then
            local cat = data.category

            if not cat or cat == 0 then
                hasUncategorized = true
            else
                local leafId = cat
                local rootId = MBM_FindRootCategory(leafId)
                if rootId then
                    local root = byRoot[rootId]
                    if not root then
                        local csName, enName = MBM_GetCategoryName(rootId)
                        root = {
                            id = rootId,
                            name_cs = csName,
                            name_en = enName,
                            total = 0,
                            subsMap = {}
                        }
                        byRoot[rootId] = root
                    end

                    root.total = root.total + (data.total or 0)

                    if leafId ~= rootId then
                        root.subsMap[leafId] = true
                    end
                end
            end
        end
    end

    local list = {}

    local rootIds = {}
    for rootId in pairs(byRoot) do
        table.insert(rootIds, rootId)
    end
    table.sort(rootIds)

    for _, rootId in ipairs(rootIds) do
        local root = byRoot[rootId]
        local cat = {
            id = rootId,
            name_cs = root.name_cs,
            name_en = root.name_en,
            subs = {}
        }

        local subIds = {}
        for subId in pairs(root.subsMap) do
            table.insert(subIds, subId)
        end
        table.sort(subIds)

        for _, subId in ipairs(subIds) do
            local csName, enName = MBM_GetCategoryName(subId)
            table.insert(
                cat.subs,
                {
                    id = subId,
                    name_cs = csName,
                    name_en = enName
                }
            )
        end

        table.insert(list, cat)
    end

    if hasUncategorized then
        table.insert(
            list,
            1,
            {
                id = 0,
                name_cs = "Nezařazeno",
                name_en = "Uncategorized",
                subs = {}
            }
        )
    end

    return list
end

-------------------------------------------------
-- Search bez zvolené kategorie (withdraw)
-------------------------------------------------
local function MBM_GetAllBankItems()
    local charCache = MBM_GetCharCache()
    local items = {}

    for itemId, data in pairs(charCache) do
        if type(data) == "table" and data.total and data.total > 0 then
            table.insert(
                items,
                {
                    id = itemId,
                    total = data.total,
                    name = data.name,
                    category = data.category
                }
            )
        end
    end

    table.sort(
        items,
        function(a, b)
            return a.id < b.id
        end
    )
    return items
end

-------------------------------------------------
-- Itemy z banky (withdraw)
-------------------------------------------------
local function MBM_GetBankItemsForCategory(selectedId)
    if not selectedId then
        return {}
    end

    local charCache = MBM_GetCharCache()
    local items = {}

    if selectedId == 0 then
        for itemId, data in pairs(charCache) do
            if type(data) == "table" and data.total and data.total > 0 then
                local cat = data.category
                if not cat or cat == 0 then
                    table.insert(
                        items,
                        {
                            id = itemId,
                            total = data.total,
                            name = data.name
                        }
                    )
                end
            end
        end

        table.sort(
            items,
            function(a, b)
                return a.id < b.id
            end
        )
        return items
    end

    local info = MBM_Categories[selectedId]
    local mode
    local rootId
    local leafId

    if info and (info.parent == 0 or info.parent == nil) then
        mode = "root"
        rootId = selectedId
    else
        mode = "leaf"
        leafId = selectedId
    end

    for itemId, data in pairs(charCache) do
        if type(data) == "table" and data.category and data.total and data.total > 0 then
            local itemCat = data.category
            local ok = false

            if mode == "leaf" then
                ok = (itemCat == leafId)
            else
                local r = MBM_FindRootCategory(itemCat)
                ok = (r == rootId)
            end

            if ok then
                table.insert(
                    items,
                    {
                        id = itemId,
                        total = data.total,
                        name = data.name
                    }
                )
            end
        end
    end

    table.sort(
        items,
        function(a, b)
            return a.id < b.id
        end
    )
    return items
end

-------------------------------------------------
-- Itemy z inventáře (deposit)
-------------------------------------------------
local function MBM_GetBagItemsForCategory(selectedId)
    if not selectedId or selectedId ~= 0 then
        return {}
    end

    local items = MBM_BuildBagItemList()

    if not MBM_GetShowHiddenDeposit() then
        local out = {}
        for _, it in ipairs(items) do
            if not MBM_IsDepositHidden(it.id) then
                table.insert(out, it)
            end
        end
        return out
    end

    return items
end

local function MBM_GetSearchQuery()
    if not MBM_SearchBox then
        return ""
    end
    local t = MBM_SearchBox:GetText() or ""
    t = string.gsub(t, "^%s+", "")
    t = string.gsub(t, "%s+$", "")
    return t
end

local function MBM_FilterItemsBySearch(items)
    local q = MBM_GetSearchQuery()
    if q == "" then
        return items
    end

    local qLower = string.lower(q)
    local qNum = tonumber(q)

    local out = {}
    for _, it in ipairs(items) do
        local id = it.id
        local name = it.name

        if not name then
            local itemName = GetItemInfo(id)
            name = itemName
        end

        local ok = false

        if qNum and id == qNum then
            ok = true
        end

        if not ok and name and name ~= "" then
            local nLower = string.lower(name)
            if string.find(nLower, qLower, 1, true) then
                ok = true
            end
        end

        if ok then
            table.insert(out, it)
        end
    end

    return out
end

-------------------------------------------------
-- Wrapper: vybere správný zdroj podle módu
-------------------------------------------------
local function MBM_GetItemsForCategory(selectedId)
    local mode = MBM_SelectedMode or MBM_GetModeSV() or "withdraw"
    if mode == "deposit" then
        return MBM_GetBagItemsForCategory(selectedId)
    else
        local items = MBM_GetBankItemsForCategory(selectedId)
        return MBM_FilterItemsBySearch(items)
    end
end

-------------------------------------------------
-- Minimap button – pozice kolem minimapy
-------------------------------------------------
local MBM_RADIUS = 82

local function MBM_UpdateMinimapPosition()
    if not MBM_MinimapButton or not Minimap then
        return
    end

    local sv = MBM_GetMinimapSV()
    local angle = sv.angle or 210

    local x = math.cos(math.rad(angle)) * MBM_RADIUS
    local y = math.sin(math.rad(angle)) * MBM_RADIUS

    MBM_MinimapButton:ClearAllPoints()
    MBM_MinimapButton:SetPoint("CENTER", Minimap, "CENTER", x, y)
end

-------------------------------------------------
-- Minimap button – vytvoření
-------------------------------------------------
local function MBM_CreateMinimapButton()
    if MBM_MinimapButton or not Minimap then
        return
    end

    local btn = CreateFrame("Button", "MaterialBank_MinimapButton", Minimap)
    MBM_MinimapButton = btn

    btn:SetSize(32, 32)
    btn:SetFrameStrata("LOW")
    btn:SetFrameLevel(8)
    btn:EnableMouse(true)

    local overlay = btn:CreateTexture(nil, "OVERLAY")
    overlay:SetTexture("Interface\\AddOns\\MaterialBank\\Textures\\border")
    overlay:SetSize(32, 32)
    overlay:SetPoint("CENTER")

    local icon = btn:CreateTexture(nil, "ARTWORK")
    icon:SetTexture("Interface\\AddOns\\MaterialBank\\Textures\\mb_icon")
    icon:SetTexCoord(0, 1, 0, 1)
    icon:SetSize(26, 26)
    icon:ClearAllPoints()
	icon:SetPoint("CENTER", btn, "CENTER", 0, -1)
    btn.icon = icon

    btn:SetHighlightTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight")

    btn:RegisterForClicks("LeftButtonUp")
    btn:RegisterForDrag("LeftButton")

    btn:SetScript(
        "OnDragStart",
        function(self)
            self:SetScript(
                "OnUpdate",
                function(self)
                    if not Minimap then
                        return
                    end

                    local mx, my = Minimap:GetCenter()
                    local cx, cy = GetCursorPosition()
                    local scale = UIParent:GetEffectiveScale()
                    cx, cy = cx / scale, cy / scale

                    local angle = math.deg(math.atan2(cy - my, cx - mx))
                    if angle < 0 then
                        angle = angle + 360
                    end

                    local sv = MBM_GetMinimapSV()
                    sv.angle = angle

                    MBM_UpdateMinimapPosition()
                end
            )
        end
    )

    btn:SetScript(
        "OnDragStop",
        function(self)
            self:SetScript("OnUpdate", nil)
        end
    )

    btn:SetScript(
        "OnEnter",
        function(self)
            GameTooltip:SetOwner(self, "ANCHOR_LEFT")
            GameTooltip:AddLine("Material Bank", 1, 1, 1)
            GameTooltip:AddLine(
                L("Kliknutím otevřeš okno materiálové banky.", "Click to open material bank window."),
                0.9,
                0.9,
                0.9,
                true
            )
            GameTooltip:Show()
        end
    )
    btn:SetScript(
        "OnLeave",
        function()
            GameTooltip:Hide()
        end
    )

    btn:SetScript(
        "OnClick",
        function(self, button)
            if button == "LeftButton" then
                if MBM_MainFrame and MBM_MainFrame:IsShown() then
                    MBM_MainFrame:Hide()
                    MBM_OpenFromMinimap = false
                else
                    if not MBM_MainFrame then
                        MBM_CreateMainWindow()
                    end
                    MBM_MainFrame:Show()
                    MBM_OpenFromMinimap = true
                end
            end
        end
    )

    MBM_UpdateMinimapPosition()
end

-------------------------------------------------
-- Kategorie – zvýraznění (active/unactive textura)
-------------------------------------------------
local function MBM_UpdateCategoryHighlights()
    local selected = MBM_SelectedCategory
    local selectedRoot = selected and MBM_FindRootCategory(selected) or nil

    for _, btn in ipairs(MBM_CategoryButtons) do
        local active = false

        if btn._isRoot then
            local st = MBM_CategoryState[btn._catId]
            if (st and st.expanded) then
                active = true
            end
            if selectedRoot and btn._catId == selectedRoot then
                active = true
            end
        else
            active = (selected ~= nil and btn._catId == selected)
        end

        MBM_SetCategoryButtonActive(btn, active)
    end
end

-------------------------------------------------
-- Kategorie – vykreslení do scroll framu (textured)
-------------------------------------------------
MBM_BuildCategoryList = function()
    MBM_ApplyLocaleStaticTexts()
    if not MBM_MainFrame or not MBM_CategoryContent then
        return
    end

    for _, btn in ipairs(MBM_CategoryButtons) do
        if btn and btn.Hide then btn:Hide() end
    end
    wipe(MBM_CategoryButtons)

    local categories = MBM_GetCategories()
    local parent = MBM_CategoryContent

    local mainCfg = MBM_CATBTN_CFG.main
    local subCfg  = MBM_CATBTN_CFG.sub

    local y = (mainCfg.y or 0)

    for _, cat in ipairs(categories) do
        local btn = CreateFrame("Button", nil, parent)
        MBM_SkinCategoryButton(btn, false)

        btn:ClearAllPoints()
        btn:SetPoint("TOPLEFT", parent, "TOPLEFT", (mainCfg.x or -6), -y)

        btn._catId = cat.id
        btn._isRoot = true
        btn._catData = cat

        if btn._mbmLabel then
            btn._mbmLabel:SetText(L(cat.name_cs, cat.name_en))
        end

        btn:SetScript("OnClick", function(self)
            wipe(MBM_AmountRemember)
            local clickedId = self._catId

            for id, st in pairs(MBM_CategoryState) do
                if id ~= clickedId then
                    st.expanded = false
                end
            end

            local st = MBM_CategoryState[clickedId] or {expanded = false}
            st.expanded = not st.expanded
            MBM_CategoryState[clickedId] = st

            if st.expanded then
                MBM_SelectedCategory = clickedId
            else
                MBM_SelectedCategory = nil
            end

            MBM_BuildCategoryList()
            MBM_RebuildItemList()
        end)

        table.insert(MBM_CategoryButtons, btn)

        y = y + (mainCfg.h or 24)

        local st = MBM_CategoryState[cat.id]
        if st and st.expanded and cat.subs and #cat.subs > 0 then
            y = y + (subCfg.startGap or 0)

            for i, sub in ipairs(cat.subs) do
                local sbtn = CreateFrame("Button", nil, parent)
                MBM_SkinCategoryButton(sbtn, true)

                sbtn:ClearAllPoints()
                sbtn:SetPoint(
                    "TOPLEFT",
                    parent,
                    "TOPLEFT",
                    (mainCfg.x or -6) + (subCfg.indentX or 0),
                    -y
                )

                sbtn._catId = sub.id
                sbtn._isRoot = false
                sbtn._parentId = cat.id
                sbtn._catData = sub

                if sbtn._mbmLabel then
                    sbtn._mbmLabel:SetText(L(sub.name_cs, sub.name_en))
                end

                sbtn:SetScript("OnClick", function(self)
                    wipe(MBM_AmountRemember)
                    MBM_SelectedCategory = self._catId
                    MBM_UpdateCategoryHighlights()
                    MBM_RebuildItemList()
                end)

                table.insert(MBM_CategoryButtons, sbtn)

                y = y + (subCfg.h or 16) + (subCfg.gap or 0)
            end

            y = y + (subCfg.afterGap or 0)
        else
            y = y + (mainCfg.gap or 0)
        end
    end

    parent:SetHeight(y + 4)
    MBM_UpdateCategoryHighlights()

    if MBM_CategoryScrollFrame then
        MBM_UpdateScrollState(MBM_CategoryScrollFrame)
    end
end

-------------------------------------------------
-- Grid nastavení pro pravý panel (items)
-------------------------------------------------
local MBM_GRID_COLS = 3
local MBM_GRID_CELL_H = 55
local MBM_GRID_GAP_X = 6
local MBM_GRID_GAP_Y = 6
local MBM_GRID_MARGIN_X = 4
local MBM_GRID_MARGIN_Y = 4

-------------------------------------------------
-- Pravý panel – seznam itemů (grid 3×N)
-------------------------------------------------
MBM_RebuildItemList = function()
    MBM_ApplyLocaleStaticTexts()
    if not MBM_ItemContent then
        return
    end

    for _, row in ipairs(MBM_ItemRows) do
        row:Hide()
    end
    wipe(MBM_ItemRows)

    local mode = MBM_SelectedMode or MBM_GetModeSV() or "withdraw"

    local items = {}

    if mode == "withdraw" then
        if MBM_SelectedCategory then
            items = MBM_GetItemsForCategory(MBM_SelectedCategory)
        else
            if MBM_GetSearchQuery() ~= "" then
                items = MBM_FilterItemsBySearch(MBM_GetAllBankItems())
            else
                MBM_ItemContent:SetHeight(4)
                if MBM_ItemScrollFrame then
                    MBM_UpdateScrollState(MBM_ItemScrollFrame)
                end
                return
            end
        end
    else
        if not MBM_SelectedCategory then
            MBM_ItemContent:SetHeight(4)
            if MBM_ItemScrollFrame then
                MBM_UpdateScrollState(MBM_ItemScrollFrame)
            end
            return
        end

        items = MBM_GetItemsForCategory(MBM_SelectedCategory)
    end

    if #items == 0 then
        local row = CreateFrame("Frame", nil, MBM_ItemContent)
        row:SetSize(MBM_ItemContent:GetWidth() - 8, 18)
        row:SetPoint("TOPLEFT", MBM_ItemContent, "TOPLEFT", 4, -4)

        local fs = row:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        fs:SetPoint("LEFT", row, "LEFT", 0, 0)
        SetFsFont(fs, 11)

        local emptyText = L("Žádné předměty v této kategorii.", "No items in this category.")
        if mode == "withdraw" and not MBM_SelectedCategory and MBM_GetSearchQuery() ~= "" then
            emptyText = L("Nic nenalezeno.", "No results.")
        end
        fs:SetText(emptyText)

        table.insert(MBM_ItemRows, row)
        MBM_ItemContent:SetHeight(24)
        if MBM_ItemScrollFrame then
            MBM_UpdateScrollState(MBM_ItemScrollFrame)
        end
        return
    end

    local contentWidth = MBM_ItemContent:GetWidth()
    local usableWidth = contentWidth - (MBM_GRID_MARGIN_X * 2)
    if usableWidth < (MBM_GRID_COLS * 32) then
        usableWidth = MBM_GRID_COLS * 32
    end

    local totalGapX = MBM_GRID_GAP_X * (MBM_GRID_COLS - 1)
    local cellW = math.floor((usableWidth - totalGapX) / MBM_GRID_COLS)

    local maxRow = 0

    for index, info in ipairs(items) do
        local idx = index - 1
        local col = idx % MBM_GRID_COLS
        local row = math.floor(idx / MBM_GRID_COLS)

        if row > maxRow then
            maxRow = row
        end

        local offsetX = MBM_GRID_MARGIN_X + col * (cellW + MBM_GRID_GAP_X)
        local offsetY = MBM_GRID_MARGIN_Y + row * (MBM_GRID_CELL_H + MBM_GRID_GAP_Y)

        local cell = CreateFrame("Button", nil, MBM_ItemContent)
        cell:SetSize(cellW, MBM_GRID_CELL_H)
        cell:SetPoint("TOPLEFT", MBM_ItemContent, "TOPLEFT", offsetX, -offsetY)
        cell:SetScript(
            "OnMouseDown",
            function()
                MBM_ClearAnyEditFocus()
            end
        )

        cell:SetBackdrop(
            {
                bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
                edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
                tile = true,
                tileSize = 16,
                edgeSize = 12,
                insets = {left = 3, right = 3, top = 3, bottom = 3}
            }
        )
        cell:SetBackdropColor(0, 0, 0, 0.8)

        local cacheName = info.name
        local itemName, _, itemQuality, _, _, _, _, _, _, itemIcon = GetItemInfo(info.id)
        local iconTex = itemIcon or GetItemIcon(info.id) or "Interface\\Icons\\INV_Misc_QuestionMark"

        if not itemName and not cacheName then
            MBM_ItemInfoPending[info.id] = true
        else
            MBM_ItemInfoPending[info.id] = nil
        end

        local icon = cell:CreateTexture(nil, "ARTWORK")
        icon:SetSize(28, 28)
        icon:SetPoint("TOPLEFT", cell, "TOPLEFT", 4, -4)
        icon:SetTexture(iconTex)

        local nameFS = cell:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
        nameFS:SetJustifyH("LEFT")
        nameFS:SetJustifyV("TOP")
        SetFsFont(nameFS, 10)

        local hideCB
        if mode == "deposit" then
            hideCB = CreateFrame("CheckButton", nil, cell, "UICheckButtonTemplate")
            hideCB:SetSize(16, 16)
            hideCB:SetPoint("TOPRIGHT", cell, "TOPRIGHT", -6, -6)

            local isHidden = MBM_IsDepositHidden(info.id)
            hideCB:SetChecked(isHidden)

            hideCB:SetScript(
                "OnClick",
                function(self)
                    MBM_ClearAnyEditFocus()
                    MBM_SetDepositHidden(info.id, self:GetChecked())

                    MBM_RebuildItemList()
                end
            )

            hideCB:SetScript(
                "OnEnter",
                function(self)
                    GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                    GameTooltip:AddLine(L("Skryté", "Hidden"), 1, 1, 1)
                    GameTooltip:AddLine(
                        L("Zaškrtni = skrýt v seznamu vkladu.", "Checked = hide from deposit list."),
                        0.9,
                        0.9,
                        0.9,
                        true
                    )
                    GameTooltip:Show()
                end
            )
            hideCB:SetScript(
                "OnLeave",
                function()
                    GameTooltip:Hide()
                end
            )
        end

        nameFS:ClearAllPoints()
        nameFS:SetPoint("TOPLEFT", icon, "TOPRIGHT", 4, -2)
        if hideCB then
            nameFS:SetPoint("RIGHT", hideCB, "LEFT", -4, 0)
        else
            nameFS:SetPoint("RIGHT", cell, "RIGHT", -4, 0)
        end

        local label = itemName or cacheName or ("Item " .. info.id)
        nameFS:SetText(label)

        if mode == "deposit" and MBM_IsDepositHidden(info.id) then
			nameFS:SetTextColor(0.65, 0.65, 0.65, 1)
			icon:SetVertexColor(0.75, 0.75, 0.75)
		else
			if itemQuality then
				local r, g, b = GetItemQualityColor(itemQuality)
				nameFS:SetTextColor(r, g, b, 1)
			else
				nameFS:SetTextColor(1, 1, 1, 1)
			end
			icon:SetVertexColor(1, 1, 1)
		end

        local countFS = cell:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
		countFS:ClearAllPoints()
		
		countFS:SetPoint("TOPLEFT", nameFS, "BOTTOMLEFT", 43, -4)
		countFS:SetPoint("RIGHT", nameFS, "RIGHT", 0, 0)
		
		countFS:SetJustifyH("LEFT")
		SetFsFont(countFS, 10)
		countFS:SetTextColor(MBM_COUNT_R, MBM_COUNT_G, MBM_COUNT_B, MBM_COUNT_A)
		countFS:SetText((info.total or 0) .. "x")
		
		if mode == "deposit" and MBM_IsDepositHidden(info.id) then
			countFS:SetTextColor(0.65, 0.65, 0.65, 1)
		else
			countFS:SetTextColor(MBM_COUNT_R, MBM_COUNT_G, MBM_COUNT_B, MBM_COUNT_A)
		end

        MBM_UniqueEditBoxId = MBM_UniqueEditBoxId + 1
        local amountBoxName = "MBM_AmountBox" .. MBM_UniqueEditBoxId

        local amountBox = CreateFrame("EditBox", amountBoxName, cell, "InputBoxTemplate")
        amountBox:SetFrameLevel(cell:GetFrameLevel() + 5)

        amountBox:SetScript(
            "OnEditFocusGained",
            function(self)
                MBM_ActiveEditBox = self
            end
        )

        amountBox:SetScript(
            "OnEditFocusLost",
            function(self)
                if MBM_ActiveEditBox == self then
                    MBM_ActiveEditBox = nil
                end
            end
        )

        amountBox:SetScript(
            "OnEscapePressed",
            function(self)
                self:ClearFocus()
            end
        )

        amountBox:SetScript(
            "OnEnterPressed",
            function(self)
                self:ClearFocus()
            end
        )

        amountBox:SetSize(34, 20)
        amountBox:SetPoint("BOTTOM", cell, "BOTTOM", -15, 6)
        amountBox:SetAutoFocus(false)
        amountBox:SetNumeric(true)
        amountBox:SetMaxLetters(4)

        local rememberKey = (mode or "withdraw") .. ":" .. tostring(info.id)

        local remembered = MBM_AmountRemember[rememberKey]
        if remembered and remembered > 0 then
            if info.total and remembered > info.total then
                remembered = info.total
                MBM_AmountRemember[rememberKey] = remembered
            end
            amountBox:SetText(tostring(remembered))
        else
			if mode == "withdraw" then
				amountBox:SetText(tostring(info.total or 1))
			else
				local def = tonumber(info.total) or 1
				if def < 1 then def = 1 end
				if def > 9999 then def = 9999 end
				amountBox:SetText(tostring(def))
			end
		end

		amountBox:SetJustifyH("CENTER")
		if amountBox.SetTextInsets then
			amountBox:SetTextInsets(-8, 0, 3, 0)
		end

        amountBox:SetScript(
            "OnTextChanged",
            function(self, userInput)
                if not userInput then
                    return
                end
                local n = tonumber(self:GetText() or "")
                if n and n > 0 then
                    MBM_AmountRemember[rememberKey] = n
                else
                    MBM_AmountRemember[rememberKey] = nil
                end
            end
        )

        local actionBtn = CreateFrame("Button", nil, cell, "UIPanelButtonTemplate")
        actionBtn:SetSize(60, 20)
        actionBtn:SetPoint("BOTTOMRIGHT", cell, "BOTTOMRIGHT", -4, 4)

        local btnLabel
        if mode == "deposit" then
            btnLabel = L("Vložit", "Deposit")
        else
            btnLabel = L("Vybrat", "Withdraw")
        end
        actionBtn:SetText(btnLabel)
        SetFsFont(actionBtn:GetFontString(), 10)

        actionBtn:SetScript(
            "OnClick",
            function()
                MBM_ClearAnyEditFocus()

                local text = amountBox:GetText()
                local n = tonumber(text)

                if not n or n <= 0 then
                    n = 1
                end

                if info.total and n > info.total then
                    n = info.total
                end

                local cmd
                if mode == "deposit" then
                    cmd = string.format(".mb push %d:%d", info.id, n)
                else
                    cmd = string.format(".mb pull %d:%d", info.id, n)
                end

                SendChatMessage(cmd, "SAY")

                if MBM_RequestHardReload then
                    MBM_RequestHardReload()
                end
            end
        )

        cell:SetScript(
            "OnEnter",
            function(self)
                GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
                GameTooltip:SetHyperlink("item:" .. info.id)
                GameTooltip:Show()
            end
        )
        cell:SetScript(
            "OnLeave",
            function()
                GameTooltip:Hide()
            end
        )
        cell:SetScript(
            "OnClick",
            function(self, button)
                if IsModifiedClick() then
                    HandleModifiedItemClick("item:" .. info.id)
                end
            end
        )

        table.insert(MBM_ItemRows, cell)
    end

    local totalRows = maxRow + 1
    local totalHeight = MBM_GRID_MARGIN_Y * 2 + totalRows * MBM_GRID_CELL_H + (totalRows - 1) * MBM_GRID_GAP_Y

    MBM_ItemContent:SetHeight(totalHeight)
    if MBM_ItemScrollFrame then
        MBM_UpdateScrollState(MBM_ItemScrollFrame)
    end
end

local function MBM_ParseMoneyBoxes(gBox, sBox, cBox)
    local g = tonumber(gBox and gBox:GetText() or "") or 0
    local s = tonumber(sBox and sBox:GetText() or "") or 0
    local c = tonumber(cBox and cBox:GetText() or "") or 0

    if g < 0 then g = 0 end
    if s < 0 then s = 0 end
    if c < 0 then c = 0 end

    if s > 99 then s = 99 end
    if c > 99 then c = 99 end

    return g, s, c
end

local function MBM_MoneyToCopper(g, s, c)
    return (g or 0) * 10000 + (s or 0) * 100 + (c or 0)
end

MBM_UpdateTopStripDepositControls = function()
    if not MBM_MainFrame or not MBM_MainFrame.topStrip then
        return
    end

    local mode = MBM_SelectedMode or MBM_GetModeSV() or "withdraw"

    -- SEARCH BOX
    if MBM_SearchBox then
        if mode == "withdraw" then
            MBM_SearchBox:Show()

            if MBM_SearchClearBtn then
                if MBM_GetSearchQuery() ~= "" then
                    MBM_SearchClearBtn:Show()
                else
                    MBM_SearchClearBtn:Hide()
                end
            end

            if MBM_SearchHint then
                if MBM_GetSearchQuery() == "" and not MBM_SearchBox:HasFocus() then
                    MBM_SearchHint:Show()
                else
                    MBM_SearchHint:Hide()
                end
            end
        else
            MBM_SearchBox:Hide()
            MBM_SearchBox:ClearFocus()
            if MBM_SearchHint then
                MBM_SearchHint:Hide()
            end
            if MBM_SearchClearBtn then
                MBM_SearchClearBtn:Hide()
            end
        end
    end

    -- DEPOSIT ALL BTN
    if MBM_DepositAllBtn then
        if mode == "deposit" then
            MBM_DepositAllBtn:Show()
        else
            MBM_DepositAllBtn:Hide()
            if MBM_DepositAllBtn._mbmLabel then
                MBM_DepositAllBtn._mbmLabel:SetText(L("Uložit vše", "Deposit all"))
            else
                MBM_DepositAllBtn:SetText(L("Uložit vše", "Deposit all"))
            end
            MBM_DepositAllBtn:Enable()
        end
    end

    -- SHOW HIDDEN
    if MBM_MainFrame.showHiddenLabel and MBM_MainFrame.showHiddenCB then
        if mode == "deposit" then
            MBM_MainFrame.showHiddenLabel:Show()
            MBM_MainFrame.showHiddenCB:Show()
            MBM_MainFrame.showHiddenCB:SetChecked(MBM_GetShowHiddenDeposit())
        else
            MBM_MainFrame.showHiddenLabel:Hide()
            MBM_MainFrame.showHiddenCB:Hide()
        end
    end

    -------------------------------------------------
    -- MONEY UI: DEPOSIT vs WITHDRAW
    -------------------------------------------------
	if mode == "deposit" then
		if MBM_DepositMoneyGoldBox   then MBM_DepositMoneyGoldBox:Show()   end
		if MBM_DepositMoneySilverBox then MBM_DepositMoneySilverBox:Show() end
		if MBM_DepositMoneyCopperBox then MBM_DepositMoneyCopperBox:Show() end
		if MBM_DepositMoneyButton    then MBM_DepositMoneyButton:Show()    end
	
		if MBM_DepositMoneyGoldLabel   then MBM_DepositMoneyGoldLabel:Show()   end
		if MBM_DepositMoneySilverLabel then MBM_DepositMoneySilverLabel:Show() end
		if MBM_DepositMoneyCopperLabel then MBM_DepositMoneyCopperLabel:Show() end
	
		if MBM_StoredMoneyText then
			local invCopper = GetMoney() or 0
			local label = L("Inventář: ", "Inventory: ")
			local money = MBM_FormatMoneyColored(invCopper)
			
			MBM_StoredMoneyText:SetText(
				MBM_ColorText(label, MBM_MONEY_LABEL_R, MBM_MONEY_LABEL_G, MBM_MONEY_LABEL_B) ..
				money
			)
			MBM_StoredMoneyText:Show()
		end
	
		if MBM_WithdrawMoneyGoldBox    then MBM_WithdrawMoneyGoldBox:Hide()    end
		if MBM_WithdrawMoneySilverBox  then MBM_WithdrawMoneySilverBox:Hide()  end
		if MBM_WithdrawMoneyCopperBox  then MBM_WithdrawMoneyCopperBox:Hide()  end
		if MBM_WithdrawMoneyButton     then MBM_WithdrawMoneyButton:Hide()     end
	
		if MBM_WithdrawMoneyGoldLabel   then MBM_WithdrawMoneyGoldLabel:Hide()   end
		if MBM_WithdrawMoneySilverLabel then MBM_WithdrawMoneySilverLabel:Hide() end
		if MBM_WithdrawMoneyCopperLabel then MBM_WithdrawMoneyCopperLabel:Hide() end
	
	else
		if MBM_DepositMoneyGoldBox   then MBM_DepositMoneyGoldBox:Hide()   end
		if MBM_DepositMoneySilverBox then MBM_DepositMoneySilverBox:Hide() end
		if MBM_DepositMoneyCopperBox then MBM_DepositMoneyCopperBox:Hide() end
		if MBM_DepositMoneyButton    then MBM_DepositMoneyButton:Hide()    end
	
		if MBM_DepositMoneyGoldLabel   then MBM_DepositMoneyGoldLabel:Hide()   end
		if MBM_DepositMoneySilverLabel then MBM_DepositMoneySilverLabel:Hide() end
		if MBM_DepositMoneyCopperLabel then MBM_DepositMoneyCopperLabel:Hide() end
	
		if MBM_StoredMoneyText then
			local copper = MBM_GetStoredMoneyCopper()
			local label = L("Uskladněno: ", "Stored: ")
			local money = MBM_FormatMoneyColored(copper)
			
			MBM_StoredMoneyText:SetText(
				MBM_ColorText(label, MBM_MONEY_LABEL_R, MBM_MONEY_LABEL_G, MBM_MONEY_LABEL_B) ..
				money
			)
			MBM_StoredMoneyText:Show()
		end
	
		if MBM_WithdrawMoneyGoldBox   then MBM_WithdrawMoneyGoldBox:Show()   end
		if MBM_WithdrawMoneySilverBox then MBM_WithdrawMoneySilverBox:Show() end
		if MBM_WithdrawMoneyCopperBox then MBM_WithdrawMoneyCopperBox:Show() end
		if MBM_WithdrawMoneyButton    then MBM_WithdrawMoneyButton:Show()    end
	
		if MBM_WithdrawMoneyGoldLabel   then MBM_WithdrawMoneyGoldLabel:Show()   end
		if MBM_WithdrawMoneySilverLabel then MBM_WithdrawMoneySilverLabel:Show() end
		if MBM_WithdrawMoneyCopperLabel then MBM_WithdrawMoneyCopperLabel:Show() end
	end
end

-------------------------------------------------
-- Deposit: ULOŽIT VŠE (s respektováním skrytých itemů)
-------------------------------------------------
MBM_DepositAllFrame = CreateFrame("Frame")
MBM_DepositAllFrame:Hide()
MBM_DepositAllFrame._queue = nil
MBM_DepositAllFrame._t = 0
MBM_DepositAllFrame._interval = 0.50

local function MBM_BuildDepositAllQueue()
    local list = MBM_BuildBagItemList()
    local q = {}

    for _, it in ipairs(list) do
        if it.id and it.total and it.total > 0 then
            if not MBM_IsDepositHidden(it.id) then
                table.insert(q, {id = it.id, n = it.total})
            end
        end
    end

    return q
end

MBM_StartDepositAll = function()
    local mode = MBM_SelectedMode or MBM_GetModeSV() or "withdraw"
    if mode ~= "deposit" then
        return
    end

    local q = MBM_BuildDepositAllQueue()
    if #q == 0 then
        DEFAULT_CHAT_FRAME:AddMessage(
            L(
                "|cffff8800[MB]|r Nemáš nic k uložení (nebo je vše skryté).",
                "|cffff8800[MB]|r Nothing to deposit (or everything is hidden)."
            )
        )
        return
    end

    MBM_DepositAllFrame._queue = q
    MBM_DepositAllFrame._t = 0
    MBM_DepositAllFrame:Show()

    if MBM_DepositAllBtn then
        MBM_DepositAllBtn:Disable()
        if MBM_DepositAllBtn._mbmLabel then
			MBM_DepositAllBtn._mbmLabel:SetText(L("Ukládám...", "Depositing..."))
		else
			MBM_DepositAllBtn:SetText(L("Ukládám...", "Depositing..."))
		end
    end
end

MBM_DepositAllFrame:SetScript(
    "OnUpdate",
    function(self, elapsed)
        if not self._queue then
            self:Hide()
            return
        end

        self._t = (self._t or 0) + (elapsed or 0)
        if self._t < (self._interval or 0.20) then
            return
        end
        self._t = 0

        local entry = table.remove(self._queue, 1)
        if not entry then
            self._queue = nil
            self:Hide()

            if MBM_DepositAllBtn then
                MBM_DepositAllBtn:Enable()
                if MBM_DepositAllBtn._mbmLabel then
					MBM_DepositAllBtn._mbmLabel:SetText(L("Uložit vše", "Deposit all"))
				else
					MBM_DepositAllBtn:SetText(L("Uložit vše", "Deposit all"))
				end
            end

            if MBM_RequestHardReload then
                MBM_RequestHardReload()
            end
            return
        end

        SendChatMessage(string.format(".mb push %d:%d", entry.id, entry.n), "SAY")
    end
)

-------------------------------------------------
-- Titulek (text uprostřed)
-------------------------------------------------
local MBM_TITLE_CFG = {
    text_cs = "Sklad materiálů",
    text_en = "Material Storage",
    fontSize = 20,
    fontFlags = "OUTLINE",
    anchorPoint = "TOP",
    relPoint = "TOP",
    x = 8,
    y = -54,
    r = 0.82,
    g = 0.82,
    b = 0.82,
    a = 1.0
}

local function MBM_CreateTitleBar(parent)
    if MBM_TitleFrame or not parent then
        return
    end

    local cfg = MBM_TITLE_CFG

    local tf = CreateFrame("Frame", nil, parent)
    tf:SetSize(360, 26)
    tf:SetPoint(cfg.anchorPoint, parent, cfg.relPoint, cfg.x or 0, cfg.y or 0)

    tf:SetFrameStrata(parent:GetFrameStrata())
    local bf = parent._mbmBorderFrame or parent
    tf:SetFrameLevel((bf:GetFrameLevel() or parent:GetFrameLevel()) + 30)

    local text = tf:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    text:SetPoint("CENTER", tf, "CENTER", 0, 0)
    SetFsFont(text, cfg.fontSize or 16, cfg.fontFlags or "OUTLINE")
    text:SetTextColor(cfg.r or 1, cfg.g or 1, cfg.b or 1, cfg.a or 1)
    MBM_TitleText = text

    MBM_TitleFrame = tf

    if MBM_UpdateTitleBar then
        MBM_UpdateTitleBar()
    end
end

MBM_UpdateTitleBar = function()
    if not MBM_TitleFrame then
        return
    end
    local cfg = MBM_TITLE_CFG

    if MBM_TitleText then
        MBM_TitleText:SetText(L(cfg.text_cs or "Material Bank", cfg.text_en or "Material Bank"))
        SetFsFont(MBM_TitleText, cfg.fontSize or 16, cfg.fontFlags or "OUTLINE")
        MBM_TitleText:SetTextColor(cfg.r or 1, cfg.g or 1, cfg.b or 1, cfg.a or 1)
    end
end

-------------------------------------------------
-- MODE BUTTONS
-------------------------------------------------
local MBM_MODEBTN_CFG = {
    w = 81,
    h = 64,
    canvasW = 256,
    canvasH = 128,
    cropW = 81,
    cropH = 64,
    texActive = "Interface\\AddOns\\MaterialBank\\Textures\\button_active",
    texInactive = "Interface\\AddOns\\MaterialBank\\Textures\\button_inactive",
    fontSize = 14,
    fontFlags = "OUTLINE"
}

local function MBM_ApplyModeBtnTex(btn, texPath)
    local cfg = MBM_MODEBTN_CFG
    if not btn or not texPath then return end

    btn:SetNormalTexture(texPath)
    btn:SetPushedTexture(texPath)

    local u2 = (cfg.cropW or cfg.w) / cfg.canvasW
    local v2 = (cfg.cropH or cfg.h) / cfg.canvasH

    local nt = btn:GetNormalTexture()
    if nt then
        nt:ClearAllPoints()
        nt:SetPoint("TOPLEFT", btn, "TOPLEFT", 0, 0)
        nt:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", 0, 0)
        nt:SetTexCoord(0, u2, 0, v2)
    end

    local pt = btn:GetPushedTexture()
    if pt then
        pt:ClearAllPoints()
        pt:SetPoint("TOPLEFT", btn, "TOPLEFT", 1, -1)
        pt:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", 1, -1)
        pt:SetTexCoord(0, u2, 0, v2)
    end
end

local function MBM_SetModeBtnActive(btn, isActive)
    local cfg = MBM_MODEBTN_CFG
    MBM_ApplyModeBtnTex(btn, isActive and cfg.texActive or cfg.texInactive)
end

-------------------------------------------------
-- CATEGORY BUTTONS (root + sub) - textura + layout
-------------------------------------------------
MBM_CATBTN_CFG = {
    main = {
        -- layout
        x = 2,
        y = 0,
        gap = 1,

        -- rozměry tlačítka
        w = 132,
        h = 24,

        -- textura je na canvasu 256x32, obrázek je vlevo nahoře 132x24
        canvasW = 256,
        canvasH = 32,
        cropW = 132,
        cropH = 24,

        pushedX = 1,
        pushedY = 1,

        texActive   = "Interface\\AddOns\\MaterialBank\\Textures\\cat_button_active",
        texInactive = "Interface\\AddOns\\MaterialBank\\Textures\\cat_button_unactive",

        fontSize = 11,
        fontFlags = "OUTLINE",
        textYOffset = 0,
    },

    sub = {
        indentX  = 22,
        startGap = 1,
        gap      = 1,
        afterGap = 1,

        w = 110,
        h = 16,

        -- textura je na canvasu 128x32, obrázek je vlevo nahoře 110x16
        canvasW = 128,
        canvasH = 32,
        cropW = 110,
        cropH = 16,

        pushedX = 1,
        pushedY = -1,

        texActive   = "Interface\\AddOns\\MaterialBank\\Textures\\subcat_button_active",
        texInactive = "Interface\\AddOns\\MaterialBank\\Textures\\subcat_button_unactive",

        fontSize = 10,
        fontFlags = "OUTLINE",
        textYOffset = 0,
    }
}

local function MBM_ApplyCatBtnTex(btn, texPath, cfg)
    if not btn or not texPath or not cfg then return end

    btn:SetNormalTexture(texPath)
    btn:SetPushedTexture(texPath)

    local u2 = (cfg.cropW or cfg.w) / (cfg.canvasW or cfg.w)
    local v2 = (cfg.cropH or cfg.h) / (cfg.canvasH or cfg.h)

    local nt = btn:GetNormalTexture()
    if nt then
        nt:SetAllPoints(btn)
        nt:SetTexCoord(0, u2, 0, v2)
    end

    local pt = btn:GetPushedTexture()
    if pt then
        pt:ClearAllPoints()
        pt:SetPoint("TOPLEFT", btn, "TOPLEFT", cfg.pushedX or 1, cfg.pushedY or -1)
        pt:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", cfg.pushedX or 1, cfg.pushedY or -1)
        pt:SetTexCoord(0, u2, 0, v2)
    end
end

MBM_SkinCategoryButton = function(btn, isSub)
    if not btn then return end
    local cfg = isSub and MBM_CATBTN_CFG.sub or MBM_CATBTN_CFG.main

    btn._mbmCatCfg = cfg
    btn:SetSize(cfg.w, cfg.h)
    btn:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")

    -- text přes texturu
    if not btn._mbmLabel then
        local fs = btn:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
        fs:SetPoint("CENTER", btn, "CENTER", 0, cfg.textYOffset or 0)
        fs:SetJustifyH("CENTER")
        fs:SetJustifyV("MIDDLE")
        fs:SetTextColor(0.95, 0.95, 0.95, 1)
        btn._mbmLabel = fs
    end
    SetFsFont(btn._mbmLabel, cfg.fontSize or 11, cfg.fontFlags or "OUTLINE")

    -- default (inactive)
    MBM_ApplyCatBtnTex(btn, cfg.texInactive, cfg)
end

MBM_SetCategoryButtonActive = function(btn, isActive)
    if not btn then return end
    local cfg = btn._mbmCatCfg or MBM_CATBTN_CFG.main
    MBM_ApplyCatBtnTex(btn, (isActive and cfg.texActive or cfg.texInactive), cfg)
end

-------------------------------------------------
-- Hlavní okno
-------------------------------------------------
local function MBM_UpdateModeButtons()
    for mode, btn in pairs(MBM_ModeButtons) do
        if btn then
            if btn._mbmIsModeBtn then
                MBM_SetModeBtnActive(btn, mode == MBM_SelectedMode)
            else
                if mode == MBM_SelectedMode then
                    btn:LockHighlight()
                else
                    btn:UnlockHighlight()
                end
            end
        end
    end
end

local function MBM_OnModeClick(self, mode)
    wipe(MBM_AmountRemember)
    MBM_SelectedMode = mode
    MBM_SetModeSV(mode)
    MBM_UpdateModeButtons()
    MBM_UpdateTopStripDepositControls()

    MBM_CategoryState = {}
    MBM_SelectedCategory = nil

    if mode == "deposit" then
        MBM_SelectedCategory = 0
        MBM_CategoryState[0] = {expanded = true}
    end

    MBM_BuildCategoryList()
    MBM_RebuildItemList()
end

-- Zavírací tlačítko konfigurace
local MBM_CLOSE_CFG = {
    -- místo kliknutí (hitbox)
    btnW = 32,
    btnH = 32,
    -- velikost samotného křížku (textura)
    crossW = 32,
    crossH = 32,
    -- velikost glow/highlight (default blizzard)
    glowW = 36,
    glowH = 36,
    -- textura
    tex = "Interface\\AddOns\\MaterialBank\\Textures\\close"
}

-- DepositAll tlačítko
local MBM_DEPOSITALL_CFG = {
    w = 100,
    h = 25,

    canvasW = 128,
    canvasH = 32,
    cropW = 100,
    cropH = 25,

    tex = "Interface\\AddOns\\MaterialBank\\Textures\\storageall",
    fontSize = 12,
    fontFlags = "OUTLINE",
    textYOffset = 0
}

local function MBM_ApplyDepositAllSkin(btn)
    if not btn then return end
    local cfg = MBM_DEPOSITALL_CFG

    btn:SetSize(cfg.w, cfg.h)

    btn:SetNormalTexture(cfg.tex)
    btn:SetPushedTexture(cfg.tex)
    btn:SetDisabledTexture(cfg.tex)
    btn:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")

    local u2 = (cfg.cropW or cfg.w) / (cfg.canvasW or cfg.w)
    local v2 = (cfg.cropH or cfg.h) / (cfg.canvasH or cfg.h)

    local nt = btn:GetNormalTexture()
    if nt then
        nt:SetAllPoints(btn)
        nt:SetTexCoord(0, u2, 0, v2)
    end

    local pt = btn:GetPushedTexture()
    if pt then
        pt:ClearAllPoints()
        pt:SetPoint("TOPLEFT", btn, "TOPLEFT", 1, -1)
        pt:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", 1, -1)
        pt:SetTexCoord(0, u2, 0, v2)
    end

    local dt = btn:GetDisabledTexture()
    if dt then
        dt:SetAllPoints(btn)
        dt:SetTexCoord(0, u2, 0, v2)
        dt:SetVertexColor(1, 1, 1, 0.35)
    end

    local ht = btn:GetHighlightTexture()
    if ht then
        ht:SetAllPoints(btn)
    end

    -- Text přes texturu
    if not btn._mbmLabel then
        local fs = btn:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
        fs:SetPoint("CENTER", btn, "CENTER", 0, cfg.textYOffset or 0)
        SetFsFont(fs, cfg.fontSize or 12, cfg.fontFlags or "OUTLINE")
        fs:SetTextColor(0.95, 0.95, 0.95, 1)
        btn._mbmLabel = fs
    else
        SetFsFont(btn._mbmLabel, cfg.fontSize or 12, cfg.fontFlags or "OUTLINE")
    end
end

local function MBM_ApplyCloseButtonSkin(btn, cfg)
    if not btn or not cfg then
        return
    end

    -- hitbox
    btn:SetSize(cfg.btnW or 32, cfg.btnH or 32)

    -- normal
    local nt = btn:GetNormalTexture()
    if nt then
        nt:SetTexture(cfg.tex)
        nt:SetTexCoord(0, 1, 0, 1)
        nt:ClearAllPoints()
        nt:SetPoint("CENTER", btn, "CENTER", 0, 0)
        nt:SetSize(cfg.crossW or 32, cfg.crossH or 32)
    end

    -- pushed
    local pt = btn:GetPushedTexture()
    if pt then
        pt:SetTexture(cfg.tex)
        pt:SetTexCoord(0, 1, 0, 1)
        pt:ClearAllPoints()
        pt:SetPoint("CENTER", btn, "CENTER", 1, -1)
        pt:SetSize(cfg.crossW or 32, cfg.crossH or 32)
    end

    -- disabled
    local dt = btn:GetDisabledTexture()
    if dt then
        dt:SetTexture(cfg.tex)
        dt:SetTexCoord(0, 1, 0, 1)
        dt:ClearAllPoints()
        dt:SetPoint("CENTER", btn, "CENTER", 0, 0)
        dt:SetSize(cfg.crossW or 32, cfg.crossH or 32)
        dt:SetVertexColor(1, 1, 1, 0.35)
    end

    -- glow/highlight
    local ht = btn:GetHighlightTexture()
    if not ht then
        btn:SetHighlightTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Highlight", "ADD")
        ht = btn:GetHighlightTexture()
    end
    if ht then
        ht:ClearAllPoints()
        ht:SetPoint("CENTER", btn, "CENTER", 0, 0)
        ht:SetSize(cfg.glowW or 36, cfg.glowH or 36)
    end
end

function MBM_CreateMainWindow()
    if MBM_MainFrame then
        return
    end

    local f = CreateFrame("Frame", "MaterialBank_MainFrame", UIParent)
    MBM_MainFrame = f

    -- ESC zavře okno
    if UISpecialFrames then
        local exists = false
        for i = 1, #UISpecialFrames do
            if UISpecialFrames[i] == "MaterialBank_MainFrame" then
                exists = true
                break
            end
        end
        if not exists then
            table.insert(UISpecialFrames, "MaterialBank_MainFrame")
        end
    end

    -- celé okno nad WoW UI
    f:SetFrameStrata("FULLSCREEN_DIALOG")
    f:SetFrameLevel(100)
    f:SetToplevel(true)
    f:Raise()

    f:HookScript(
        "OnHide",
        function()
            MBM_OpenFromMinimap = false
            wipe(MBM_AmountRemember)
        end
    )
	
    f:HookScript(
        "OnShow",
        function()
            MBM_SelectedMode = MBM_GetModeSV() or MBM_SelectedMode or "withdraw"

            if MBM_SelectedMode == "deposit" then
                MBM_SelectedCategory = 0
                MBM_CategoryState = MBM_CategoryState or {}
                MBM_CategoryState[0] = { expanded = true }
            else
                MBM_SelectedCategory = nil
            end

            MBM_UpdateModeButtons()
            MBM_UpdateTopStripDepositControls()

            MBM_BuildCategoryList()
            MBM_RebuildItemList()

            if MBM_ItemScrollFrame then
                MBM_UpdateScrollState(MBM_ItemScrollFrame)
            end
            if MBM_CategoryScrollFrame then
                MBM_UpdateScrollState(MBM_CategoryScrollFrame)
            end
        end
    )

    -------------------------------------------------
    -- CUSTOM FRAME
    -------------------------------------------------
    local FRAME_W, FRAME_H = 739, 560
    local INNER_X, INNER_Y = 46, 57
    local INNER_W, INNER_H = 647, 420

    f:SetSize(FRAME_W, FRAME_H)
    f:SetScale(0.9)
    f:SetPoint("CENTER", UIParent, "CENTER", 0, 0)

    if f.SetBackdrop then
        f:SetBackdrop(nil)
    end

    -- =================================================
    -- FRAME TEXTURE
    -- =================================================
    local TEX_W, TEX_H = 1024, 1024

    local u1, u2 = 0, FRAME_W / TEX_W
    local v1, v2 = 0, FRAME_H / TEX_H

    local borderFrame = CreateFrame("Frame", nil, f)
    borderFrame:SetAllPoints(f)
    borderFrame:SetFrameStrata(f:GetFrameStrata())
    borderFrame:SetFrameLevel(f:GetFrameLevel() + 10)

    local frameTex = borderFrame:CreateTexture(nil, "OVERLAY")
    frameTex:SetAllPoints(borderFrame)
    frameTex:SetTexture("Interface\\AddOns\\MaterialBank\\Textures\\frame")
    frameTex:SetTexCoord(u1, u2, v1, v2)
    frameTex:SetAlpha(1.0)

    f._mbmBorderFrame = borderFrame
    f._mbmFrameTex = frameTex

    -- contentRoot
    local contentRoot = CreateFrame("Frame", nil, f)
    contentRoot:SetPoint("TOPLEFT", f, "TOPLEFT", INNER_X, -INNER_Y)
    contentRoot:SetSize(INNER_W, INNER_H)
    f.contentRoot = contentRoot
    contentRoot:SetFrameStrata(f:GetFrameStrata())
    contentRoot:SetFrameLevel(f:GetFrameLevel() + 1)

    -- Background jen do vnitřku
    local bg = contentRoot:CreateTexture(nil, "BACKGROUND")
    local BG_PAD_TOP = 7
    local BG_PAD_BOTTOM = 7

    bg:ClearAllPoints()
    bg:SetPoint("TOPLEFT", contentRoot, "TOPLEFT", 0, BG_PAD_TOP)
    bg:SetPoint("BOTTOMRIGHT", contentRoot, "BOTTOMRIGHT", 0, -BG_PAD_BOTTOM)

    bg:SetTexture("Interface\\AddOns\\MaterialBank\\Textures\\background.tga")
    bg:SetAlpha(1.0)
    f._mbmBg = bg

    local root = contentRoot

    MBM_CreateTitleBar(f)

    f:EnableMouse(true)
    f:SetMovable(true)
    f:RegisterForDrag("LeftButton")
    f:SetScript("OnDragStart", f.StartMoving)
    f:SetScript("OnDragStop", f.StopMovingOrSizing)
    f:HookScript(
        "OnMouseDown",
        function()
            MBM_ClearAnyEditFocus()
        end
    )

    local close = CreateFrame("Button", nil, f, "UIPanelCloseButton")
    close:ClearAllPoints()
    close:SetPoint("TOPRIGHT", f, "TOPRIGHT", -49, -55)

    close:SetFrameStrata(f:GetFrameStrata())
    local bf = f._mbmBorderFrame or f
    close:SetFrameLevel((bf:GetFrameLevel() or f:GetFrameLevel()) + 40)

    MBM_ApplyCloseButtonSkin(close, MBM_CLOSE_CFG)

    close:SetScript(
        "OnClick",
        function()
            if MBM_MainFrame then
                MBM_MainFrame:Hide()
            else
                f:Hide()
            end
        end
    )

	-- SYNC tlačítko
	local SYNC_W, SYNC_H = 60, 20
	local CANVAS_W, CANVAS_H = 64, 32
	local u2, v2 = SYNC_W / CANVAS_W, SYNC_H / CANVAS_H
	
	local syncBtn = CreateFrame("Button", nil, f)
	syncBtn:SetSize(SYNC_W, SYNC_H)
	syncBtn:SetPoint("RIGHT", root, "TOPRIGHT", -60, -12)
	
	syncBtn:SetNormalTexture("Interface\\AddOns\\MaterialBank\\Textures\\sync")
	syncBtn:SetPushedTexture("Interface\\AddOns\\MaterialBank\\Textures\\sync")
	syncBtn:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")
	
	local nt = syncBtn:GetNormalTexture()
	if nt then
		nt:SetAllPoints(syncBtn)
		nt:SetTexCoord(0, u2, 0, v2)
	end
	
	local pt = syncBtn:GetPushedTexture()
	if pt then
		pt:ClearAllPoints()
		pt:SetPoint("TOPLEFT", syncBtn, "TOPLEFT", 1, -1)
		pt:SetPoint("BOTTOMRIGHT", syncBtn, "BOTTOMRIGHT", 1, -1)
		pt:SetTexCoord(0, u2, 0, v2)
	end
	
	-- TEXT
	local txt = syncBtn:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
	txt:SetPoint("CENTER", syncBtn, "CENTER", 0, 2)
	txt:SetText(L("Sync", "Sync"))
	SetFsFont(txt, 13, "OUTLINE")
	txt:SetTextColor(0.95, 0.95, 0.95, 1)
	
	syncBtn:SetScript("OnClick", function()
		SendChatMessage(".mb sync", "SAY")
		if MBM_RequestHardReload then
			MBM_RequestHardReload()
		end
	end)

	-------------------------------------------------
	-- MODE BUTTONS
	-------------------------------------------------
	local MODE_SHIFT_Y = (MBM_MODEBTN_CFG.h - 22)
	
	local modeBar = CreateFrame("Frame", nil, root)
	modeBar:SetSize(252, 97)
	modeBar:SetPoint("TOPLEFT", root, "TOPLEFT", 2, 0)
	modeBar:SetFrameStrata(f:GetFrameStrata())
	modeBar:SetFrameLevel((root:GetFrameLevel() or 0) + 2)
	
	local withdrawBtn = CreateFrame("Button", nil, modeBar)
	withdrawBtn:SetSize(MBM_MODEBTN_CFG.w, MBM_MODEBTN_CFG.h)
	withdrawBtn:SetPoint("TOPLEFT", modeBar, "TOPLEFT", 0, 0)
	withdrawBtn:SetFrameLevel(modeBar:GetFrameLevel() + 1)
	withdrawBtn._mbmIsModeBtn = true
	
	withdrawBtn:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")
	
	local wText = withdrawBtn:CreateFontString(nil, "OVERLAY", "GameFontNormal")
	wText:SetPoint("CENTER", withdrawBtn, "CENTER", 0, 0)
	SetFsFont(wText, MBM_MODEBTN_CFG.fontSize, MBM_MODEBTN_CFG.fontFlags)
	withdrawBtn._mbmLabel = wText
	
	withdrawBtn:SetScript("OnClick", function(self) MBM_OnModeClick(self, "withdraw") end)
	
	local depositBtn = CreateFrame("Button", nil, modeBar)
	depositBtn:SetSize(MBM_MODEBTN_CFG.w, MBM_MODEBTN_CFG.h)
	depositBtn:SetPoint("TOPLEFT", withdrawBtn, "TOPRIGHT", 0, 0)
	depositBtn:SetFrameLevel(modeBar:GetFrameLevel() + 1)
	depositBtn._mbmIsModeBtn = true
	
	depositBtn:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")
	
	local dText = depositBtn:CreateFontString(nil, "OVERLAY", "GameFontNormal")
	dText:SetPoint("CENTER", depositBtn, "CENTER", 0, 0)
	SetFsFont(dText, MBM_MODEBTN_CFG.fontSize, MBM_MODEBTN_CFG.fontFlags)
	depositBtn._mbmLabel = dText
	
	depositBtn:SetScript("OnClick", function(self) MBM_OnModeClick(self, "deposit") end)
	
	MBM_ModeButtons.withdraw = withdrawBtn
	MBM_ModeButtons.deposit = depositBtn

    local leftPane = CreateFrame("Frame", nil, f)
    leftPane:SetPoint("TOPLEFT", root, "TOPLEFT", 10, -67)
    leftPane:SetPoint("BOTTOMLEFT", root, "BOTTOMLEFT", 10, 10)
    leftPane:SetWidth(150)
    f.leftPane = leftPane

    local leftBg = leftPane:CreateTexture(nil, "BACKGROUND")
    leftBg:SetAllPoints()
    leftBg:SetTexture(0, 0, 0, 0.35)
    f.leftBg = leftBg

	local scroll = CreateFrame("ScrollFrame", "MaterialBankCategoryScrollFrame", leftPane, "UIPanelScrollFrameTemplate")
	scroll:SetPoint("TOPLEFT", leftPane, "TOPLEFT", -8, -4)
	scroll:SetPoint("BOTTOMRIGHT", leftPane, "BOTTOMRIGHT", -22, -6)
	
	MBM_CategoryScrollFrame = scroll
	MBM_ApplyScrollBarSkin(scroll)
	
	scroll:EnableMouseWheel(true)
	scroll:SetScript(
		"OnMouseWheel",
		function(self, delta)
			MBM_ClearAnyEditFocus()
	
			local step = 18
			local cur = self:GetVerticalScroll() or 0
			local range = self:GetVerticalScrollRange() or 0
	
			local new = cur - (delta * step)
			if new < 0 then
				new = 0
			end
			if new > range then
				new = range
			end
	
			self:SetVerticalScroll(new)
			MBM_UpdateScrollState(self)
		end
	)
	
	local content = CreateFrame("Frame", "MaterialBankCategoryScrollChild", scroll)
	content:SetPoint("TOPLEFT", scroll, "TOPLEFT", 11, 0)
	content:SetWidth(150 - 22)
	scroll:SetScrollChild(content)
	MBM_CategoryContent = content
	
	scroll:SetScript(
		"OnSizeChanged",
		function(self, width, height)
			MBM_CategoryContent:SetWidth((width or 0) - 11)
		end
	)

    local rightPane = CreateFrame("Frame", nil, f)
    rightPane:SetPoint("TOPLEFT", root, "TOPLEFT", 10 + 150 + 8, -67)
    rightPane:SetPoint("BOTTOMRIGHT", root, "BOTTOMRIGHT", -10, 10)
    f.rightPane = rightPane

    local rightBg = rightPane:CreateTexture(nil, "BACKGROUND")
    rightBg:SetAllPoints()
    rightBg:SetTexture(0, 0, 0, 0.25)

    local itemScroll =
        CreateFrame("ScrollFrame", "MaterialBankItemScrollFrame", rightPane, "UIPanelScrollFrameTemplate")
    itemScroll:SetPoint("TOPLEFT", rightPane, "TOPLEFT", 0, -4)
    itemScroll:SetPoint("BOTTOMRIGHT", rightPane, "BOTTOMRIGHT", -22, -6)
    MBM_ItemScrollFrame = itemScroll
    MBM_ApplyScrollBarSkin(itemScroll)

    itemScroll:EnableMouseWheel(true)
    itemScroll:SetScript(
        "OnMouseWheel",
        function(self, delta)
            MBM_ClearAnyEditFocus()
            local step = (MBM_GRID_CELL_H + MBM_GRID_GAP_Y)
            local cur = self:GetVerticalScroll() or 0
            local range = self:GetVerticalScrollRange() or 0

            local new = cur - (delta * step)
            if new < 0 then
                new = 0
            end
            if new > range then
                new = range
            end

            self:SetVerticalScroll(new)
            MBM_UpdateScrollState(self)
        end
    )

    local itemContent = CreateFrame("Frame", "MaterialBankItemScrollChild", itemScroll)
    itemContent:SetPoint("TOPLEFT", itemScroll, "TOPLEFT", 0, 0)
    itemContent:SetWidth(rightPane:GetWidth() - 22)
    itemScroll:SetScrollChild(itemContent)
    MBM_ItemContent = itemContent

    itemScroll:SetScript(
        "OnSizeChanged",
        function(self, width, height)
            MBM_ItemContent:SetWidth(width)
        end
    )

    local topStrip = f:CreateTexture(nil, "BORDER")
    topStrip:SetTexture(0.1, 0.1, 0.1, 0.0)
    topStrip:SetPoint("TOPLEFT", root, "TOPLEFT", 10 + 150 + 8, -36)
    topStrip:SetPoint("TOPRIGHT", root, "TOPRIGHT", -10, -36)
    topStrip:SetHeight(24)
    f.topStrip = topStrip

    -------------------------------------------------
    -- Separátory (na popředí) + topStrip UI (uvnitř root)
    -------------------------------------------------
    local root = f.contentRoot or f

    local lines = CreateFrame("Frame", nil, root)
    lines:SetAllPoints(root)
    lines:SetFrameStrata(f:GetFrameStrata())
    lines:SetFrameLevel(f:GetFrameLevel() + 50)
    f._mbmLines = lines

    local function MakeLine()
        local t = lines:CreateTexture(nil, "OVERLAY")
        MBM_SetLineColor(t)
        return t
    end

    -------------------------------------------------
    -- Existující separátory (jen uvnitř root)
    -------------------------------------------------
    local padL, padR, padB = 1, 1, 1

    local sepH = MakeLine()
    sepH:SetPoint("TOPLEFT", root, "TOPLEFT", padL, -64)
    sepH:SetPoint("TOPRIGHT", root, "TOPRIGHT", -padR, -64)
    sepH:SetHeight(2)

    local sepV = MakeLine()
    sepV:SetPoint("TOPLEFT", root, "TOPLEFT", 10 + 150 + 4, -1)
    sepV:SetPoint("BOTTOMLEFT", root, "BOTTOMLEFT", 10 + 150 + 4, padB - 2)
    sepV:SetWidth(2)

    -------------------------------------------------
    -- showHidden label + checkbox (parent = root)
    -------------------------------------------------
    local showHiddenLabel = root:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    showHiddenLabel:SetText(L("Zobrazit skryté:", "Show hidden:"))
    SetFsFont(showHiddenLabel, 10)
    showHiddenLabel:Hide()
    f.showHiddenLabel = showHiddenLabel

    local showHiddenCB = CreateFrame("CheckButton", nil, root, "UICheckButtonTemplate")
    showHiddenCB:SetSize(20, 20)
    showHiddenCB:SetChecked(MBM_GetShowHiddenDeposit())
    showHiddenCB:Hide()
    showHiddenCB:SetScript(
        "OnClick",
        function(self)
            MBM_ClearAnyEditFocus()
            MBM_SetShowHiddenDeposit(self:GetChecked())
            MBM_RebuildItemList()
            MBM_UpdateTopStripDepositControls()
        end
    )
    f.showHiddenCB = showHiddenCB

    showHiddenCB:ClearAllPoints()
    showHiddenCB:SetPoint("RIGHT", topStrip, "RIGHT", -21, -8)

    showHiddenLabel:ClearAllPoints()
    showHiddenLabel:SetPoint("RIGHT", showHiddenCB, "LEFT", -4, 1)

    -------------------------------------------------
    -- Search box (jen withdraw) (parent = root)
    -------------------------------------------------
    local searchBox = CreateFrame("EditBox", "MaterialBank_SearchBox", root, "InputBoxTemplate")
    searchBox:SetSize(90, 18)
    searchBox:SetAutoFocus(false)
    searchBox:SetMaxLetters(40)
    searchBox:SetPoint("LEFT", topStrip, "LEFT", 12, -1)
    searchBox:Hide()
    MBM_SearchBox = searchBox
    f.searchText = searchBox

    searchBox:SetScript(
        "OnEscapePressed",
        function(self)
            self:ClearFocus()
        end
    )
    searchBox:SetScript(
        "OnEnterPressed",
        function(self)
            self:ClearFocus()
        end
    )

    searchBox:SetScript(
        "OnTextChanged",
        function(self)
            local t = MBM_GetSearchQuery()

            if MBM_SearchHint then
                if t == "" and not self:HasFocus() then
                    MBM_SearchHint:Show()
                else
                    MBM_SearchHint:Hide()
                end
            end

            if MBM_SearchClearBtn then
                if t ~= "" then
                    MBM_SearchClearBtn:Show()
                else
                    MBM_SearchClearBtn:Hide()
                end
            end

            if MBM_MainFrame and MBM_MainFrame:IsShown() then
                MBM_RebuildItemList()
            end
        end
    )

    -- “placeholder” text (fake) (parent = root)
    local hint = root:CreateFontString(nil, "OVERLAY", "GameFontDisableSmall")
    hint:SetPoint("LEFT", searchBox, "LEFT", 6, 0)
    hint:SetText(L("Hledat...", "Search..."))
    SetFsFont(hint, 10)
    hint:SetTextColor(0.85, 0.85, 0.85, 0.9)
    MBM_SearchHint = hint
    hint:Hide()

    -- Clear "X" button (parent = root)
    local clearBtn = CreateFrame("Button", nil, root, "UIPanelButtonTemplate")
    clearBtn:SetSize(18, 18)
    clearBtn:SetPoint("LEFT", searchBox, "RIGHT", -20, 19)
    clearBtn:SetText("X")
    SetFsFont(clearBtn:GetFontString(), 10)
    clearBtn:Hide()
    MBM_SearchClearBtn = clearBtn

    clearBtn:SetScript(
        "OnClick",
        function()
            MBM_ClearAnyEditFocus()
            searchBox:SetText("")
            searchBox:ClearFocus()

            if MBM_SearchHint then
                MBM_SearchHint:Show()
            end
            if MBM_MainFrame and MBM_MainFrame:IsShown() then
                MBM_RebuildItemList()
            end
        end
    )

    -------------------------------------------------
    -- Deposit: tlačítko "Uložit vše" (parent = root)
    -------------------------------------------------
	local depositAllBtn = CreateFrame("Button", nil, root)
	MBM_ApplyDepositAllSkin(depositAllBtn)
	
	-- Pozice
	depositAllBtn:SetPoint("LEFT", topStrip, "LEFT", 1, -1)
	
	depositAllBtn:Hide()
	MBM_DepositAllBtn = depositAllBtn
	
	if depositAllBtn._mbmLabel then
		depositAllBtn._mbmLabel:SetText(L("Uložit vše", "Deposit all"))
	end

    depositAllBtn:SetScript(
        "OnClick",
        function()
            MBM_ClearAnyEditFocus()
            if MBM_StartDepositAll then
                MBM_StartDepositAll()
            end
        end
    )

    depositAllBtn:SetScript(
        "OnEnter",
        function(self)
            GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
            GameTooltip:AddLine(L("Uložit vše", "Deposit all"), 1, 1, 1)
            GameTooltip:AddLine(
                L(
                    "Uloží všechny materiály z batohu.\nSkryté položky se NEULOŽÍ (funguje jako zámek).",
                    "Deposits all materials from your bags.\nHidden items will NOT be deposited (acts as a lock)."
                ),
                0.9,
                0.9,
                0.9,
                true
            )
            GameTooltip:Show()
        end
    )
    depositAllBtn:SetScript(
        "OnLeave",
        function()
            GameTooltip:Hide()
        end
    )

    searchBox:SetScript(
        "OnEditFocusGained",
        function(self)
            if MBM_SearchHint then
                MBM_SearchHint:Hide()
            end
            MBM_ActiveEditBox = self
        end
    )

    searchBox:SetScript(
        "OnEditFocusLost",
        function(self)
            if MBM_ActiveEditBox == self then
                MBM_ActiveEditBox = nil
            end
            if MBM_SearchHint then
                local t = MBM_GetSearchQuery()
                if t == "" then
                    MBM_SearchHint:Show()
                end
            end
        end
    )

    -- nastaví anchor searchText + ukáže/skryje showHidden prvky podle módu
    MBM_SelectedMode = MBM_GetModeSV() or "withdraw"
    MBM_UpdateModeButtons()
    MBM_UpdateTopStripDepositControls()
    MBM_ApplyLocaleStaticTexts()

    MBM_CategoryState = {}
    MBM_SelectedCategory = nil

    if MBM_SelectedMode == "deposit" then
        MBM_SelectedCategory = 0
        MBM_CategoryState[0] = {expanded = true}
    end

    MBM_BuildCategoryList()
    MBM_UpdateTopStripDepositControls()
    f:Hide()

    if MBM_ItemScrollFrame then
        MBM_UpdateScrollState(MBM_ItemScrollFrame)
    end

    -------------------------------------------------
    -- Money DEPOSIT (3 boxy + Vložit) + IKONY
    -------------------------------------------------

    -- mini helper
    local function MBM_MakeCoinIcon(parent, anchorTo, texPath, size, x, y)
        local t = parent:CreateTexture(nil, "OVERLAY")
        t:SetTexture(texPath)
        t:SetSize(size or 14, size or 14)
        t:SetPoint("LEFT", anchorTo, "RIGHT", x or 2, y or 0)
        t:Hide()
        return t
    end

    -- Gold box (7 cifer)
    MBM_DepositMoneyGoldBox = CreateFrame("EditBox", "MaterialBank_DepositGold", root, "InputBoxTemplate")
    MBM_DepositMoneyGoldBox:SetSize(53, 20)
    MBM_DepositMoneyGoldBox:SetAutoFocus(false)
    MBM_DepositMoneyGoldBox:SetNumeric(true)
    MBM_DepositMoneyGoldBox:SetMaxLetters(7)
    MBM_DepositMoneyGoldBox:SetPoint("LEFT", depositAllBtn, "RIGHT", 18, 0)
    MBM_DepositMoneyGoldBox:SetJustifyH("CENTER")

    -- GOLD ikona
    local depGoldIcon = MBM_MakeCoinIcon(root, MBM_DepositMoneyGoldBox, "Interface\\MoneyFrame\\UI-GoldIcon", 14, 2, 0)
    MBM_DepositMoneyGoldLabel = depGoldIcon

    -- Silver box (max 99)
    MBM_DepositMoneySilverBox = CreateFrame("EditBox", "MaterialBank_DepositSilver", root, "InputBoxTemplate")
    MBM_DepositMoneySilverBox:SetSize(20, 20)
    MBM_DepositMoneySilverBox:SetAutoFocus(false)
    MBM_DepositMoneySilverBox:SetNumeric(true)
    MBM_DepositMoneySilverBox:SetMaxLetters(2)
    MBM_DepositMoneySilverBox:SetPoint("LEFT", depGoldIcon, "RIGHT", 6, 0)
    MBM_DepositMoneySilverBox:SetJustifyH("CENTER")

    -- SILVER ikona
    local depSilverIcon = MBM_MakeCoinIcon(root, MBM_DepositMoneySilverBox, "Interface\\MoneyFrame\\UI-SilverIcon", 14, 2, 0)
    MBM_DepositMoneySilverLabel = depSilverIcon

    -- Copper box (max 99)
    MBM_DepositMoneyCopperBox = CreateFrame("EditBox", "MaterialBank_DepositCopper", root, "InputBoxTemplate")
    MBM_DepositMoneyCopperBox:SetSize(20, 20)
    MBM_DepositMoneyCopperBox:SetAutoFocus(false)
    MBM_DepositMoneyCopperBox:SetNumeric(true)
    MBM_DepositMoneyCopperBox:SetMaxLetters(2)
    MBM_DepositMoneyCopperBox:SetPoint("LEFT", depSilverIcon, "RIGHT", 6, 0)
    MBM_DepositMoneyCopperBox:SetJustifyH("CENTER")

    -- COPPER ikona
    local depCopperIcon = MBM_MakeCoinIcon(root, MBM_DepositMoneyCopperBox, "Interface\\MoneyFrame\\UI-CopperIcon", 14, 2, 0)
    MBM_DepositMoneyCopperLabel = depCopperIcon

    -- Tlačítko "Vložit"
    MBM_DepositMoneyButton = CreateFrame("Button", nil, root)
    local SYNC_W, SYNC_H = 60, 20
    local CANVAS_W, CANVAS_H = 64, 32
    local u2, v2 = SYNC_W / CANVAS_W, SYNC_H / CANVAS_H

    MBM_DepositMoneyButton:SetSize(SYNC_W, SYNC_H)
    MBM_DepositMoneyButton:SetPoint("LEFT", depCopperIcon, "RIGHT", 5.25, 0)

    MBM_DepositMoneyButton:SetNormalTexture("Interface\\AddOns\\MaterialBank\\Textures\\sync")
    MBM_DepositMoneyButton:SetPushedTexture("Interface\\AddOns\\MaterialBank\\Textures\\sync")
    MBM_DepositMoneyButton:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")

    local d_nt = MBM_DepositMoneyButton:GetNormalTexture()
    if d_nt then
        d_nt:SetAllPoints(MBM_DepositMoneyButton)
        d_nt:SetTexCoord(0, u2, 0, v2)
    end

    local d_pt = MBM_DepositMoneyButton:GetPushedTexture()
    if d_pt then
        d_pt:ClearAllPoints()
        d_pt:SetPoint("TOPLEFT", MBM_DepositMoneyButton, "TOPLEFT", 1, -1)
        d_pt:SetPoint("BOTTOMRIGHT", MBM_DepositMoneyButton, "BOTTOMRIGHT", 1, -1)
        d_pt:SetTexCoord(0, u2, 0, v2)
    end

    local d_txt = MBM_DepositMoneyButton:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    d_txt:SetPoint("CENTER", MBM_DepositMoneyButton, "CENTER", 0, 1)
    d_txt:SetText(L("Vložit", "Deposit"))
    SetFsFont(d_txt, 13, "OUTLINE")
    d_txt:SetTextColor(0.95, 0.95, 0.95, 1)

    MBM_DepositMoneyButton:SetScript("OnClick", function()
        MBM_ClearAnyEditFocus()

        local g, s, c = MBM_ParseMoneyBoxes(
            MBM_DepositMoneyGoldBox,
            MBM_DepositMoneySilverBox,
            MBM_DepositMoneyCopperBox
        )
        local totalCopper = MBM_MoneyToCopper(g, s, c)

        if totalCopper <= 0 then
            UIErrorsFrame:AddMessage(
                L("Zadej částku k uložení.", "Enter amount to deposit."),
                1, 0.2, 0.2
            )
            return
        end

        local playerMoney = GetMoney() or 0
        if totalCopper > playerMoney then
            totalCopper = playerMoney
        end

        if totalCopper <= 0 then
            UIErrorsFrame:AddMessage(
                L("Nemáš dost peněz.", "You don't have enough money."),
                1, 0.2, 0.2
            )
            return
        end

        -- PŘÍKAZ: VLOŽIT PENÍZE
        SendChatMessage(string.format(".mb money deposit %d", totalCopper), "SAY")

        local sv = MBM_GetMoneySV()
        sv.copper = math.max(0, (sv.copper or 0) + totalCopper)

        MBM_UpdateTopStripDepositControls()
    end)

    MBM_DepositMoneyGoldBox:Hide()
    MBM_DepositMoneySilverBox:Hide()
    MBM_DepositMoneyCopperBox:Hide()
    MBM_DepositMoneyButton:Hide()
    if MBM_DepositMoneyGoldLabel then MBM_DepositMoneyGoldLabel:Hide() end
    if MBM_DepositMoneySilverLabel then MBM_DepositMoneySilverLabel:Hide() end
    if MBM_DepositMoneyCopperLabel then MBM_DepositMoneyCopperLabel:Hide() end

    -------------------------------------------------
    -- Money WITHDRAW (uloženo + 3 boxy + Vybrat) + IKONY
    -------------------------------------------------
    -- Text "Uskladněno: Xg Ys Zc"
    MBM_StoredMoneyText = root:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
    MBM_StoredMoneyText:SetPoint("LEFT", topStrip, "LEFT", 112, 17)
    SetFsFont(MBM_StoredMoneyText, 11)
    MBM_StoredMoneyText:SetText("")
    MBM_StoredMoneyText:Hide()

    -- Gold (7 cifer)
    MBM_WithdrawMoneyGoldBox = CreateFrame("EditBox", "MaterialBank_WithdrawGold", root, "InputBoxTemplate")
    MBM_WithdrawMoneyGoldBox:SetSize(53, 20)
    MBM_WithdrawMoneyGoldBox:SetAutoFocus(false)
    MBM_WithdrawMoneyGoldBox:SetNumeric(true)
    MBM_WithdrawMoneyGoldBox:SetMaxLetters(6)
    MBM_WithdrawMoneyGoldBox:SetJustifyH("CENTER")

    -- Tlačítko "Vybrat"
    MBM_WithdrawMoneyButton = CreateFrame("Button", nil, root)
    local SYNC_W, SYNC_H = 60, 20
    local CANVAS_W, CANVAS_H = 64, 32
    local u2, v2 = SYNC_W / CANVAS_W, SYNC_H / CANVAS_H

    MBM_WithdrawMoneyButton:SetSize(SYNC_W, SYNC_H)
    MBM_WithdrawMoneyButton:SetPoint("RIGHT", topStrip, "RIGHT", -132, -1)

    MBM_WithdrawMoneyButton:SetNormalTexture("Interface\\AddOns\\MaterialBank\\Textures\\sync")
    MBM_WithdrawMoneyButton:SetPushedTexture("Interface\\AddOns\\MaterialBank\\Textures\\sync")
    MBM_WithdrawMoneyButton:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")

    local w_nt = MBM_WithdrawMoneyButton:GetNormalTexture()
    if w_nt then
        w_nt:SetAllPoints(MBM_WithdrawMoneyButton)
        w_nt:SetTexCoord(0, u2, 0, v2)
    end

    local w_pt = MBM_WithdrawMoneyButton:GetPushedTexture()
    if w_pt then
        w_pt:ClearAllPoints()
        w_pt:SetPoint("TOPLEFT", MBM_WithdrawMoneyButton, "TOPLEFT", 1, -1)
        w_pt:SetPoint("BOTTOMRIGHT", MBM_WithdrawMoneyButton, "BOTTOMRIGHT", 1, -1)
        w_pt:SetTexCoord(0, u2, 0, v2)
    end

    local w_txt = MBM_WithdrawMoneyButton:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
    w_txt:SetPoint("CENTER", MBM_WithdrawMoneyButton, "CENTER", 0, 1)
    w_txt:SetText(L("Vybrat", "Withdraw"))
    SetFsFont(w_txt, 13, "OUTLINE")
    w_txt:SetTextColor(0.95, 0.95, 0.95, 1)

    MBM_WithdrawMoneyGoldBox:SetPoint("RIGHT", MBM_WithdrawMoneyButton, "LEFT", -105, 0)

    -- GOLD ikona
    local wGoldIcon = MBM_MakeCoinIcon(root, MBM_WithdrawMoneyGoldBox, "Interface\\MoneyFrame\\UI-GoldIcon", 14, 2, 0)
    MBM_WithdrawMoneyGoldLabel = wGoldIcon

    MBM_WithdrawMoneySilverBox = CreateFrame("EditBox", "MaterialBank_WithdrawSilver", root, "InputBoxTemplate")
    MBM_WithdrawMoneySilverBox:SetSize(20, 20)
    MBM_WithdrawMoneySilverBox:SetAutoFocus(false)
    MBM_WithdrawMoneySilverBox:SetNumeric(true)
    MBM_WithdrawMoneySilverBox:SetMaxLetters(2)
    MBM_WithdrawMoneySilverBox:SetJustifyH("CENTER")
    MBM_WithdrawMoneySilverBox:SetPoint("LEFT", wGoldIcon, "RIGHT", 6, 0)

    -- SILVER ikona
    local wSilverIcon = MBM_MakeCoinIcon(root, MBM_WithdrawMoneySilverBox, "Interface\\MoneyFrame\\UI-SilverIcon", 14, 2, 0)
    MBM_WithdrawMoneySilverLabel = wSilverIcon

    MBM_WithdrawMoneyCopperBox = CreateFrame("EditBox", "MaterialBank_WithdrawCopper", root, "InputBoxTemplate")
    MBM_WithdrawMoneyCopperBox:SetSize(20, 20)
    MBM_WithdrawMoneyCopperBox:SetAutoFocus(false)
    MBM_WithdrawMoneyCopperBox:SetNumeric(true)
    MBM_WithdrawMoneyCopperBox:SetMaxLetters(2)
    MBM_WithdrawMoneyCopperBox:SetJustifyH("CENTER")
    MBM_WithdrawMoneyCopperBox:SetPoint("LEFT", wSilverIcon, "RIGHT", 6, 0)

    -- COPPER ikona
    local wCopperIcon = MBM_MakeCoinIcon(root, MBM_WithdrawMoneyCopperBox, "Interface\\MoneyFrame\\UI-CopperIcon", 14, 2, 0)
    MBM_WithdrawMoneyCopperLabel = wCopperIcon

    MBM_WithdrawMoneyButton:SetScript("OnClick", function()
        MBM_ClearAnyEditFocus()

        local g, s, c = MBM_ParseMoneyBoxes(
            MBM_WithdrawMoneyGoldBox,
            MBM_WithdrawMoneySilverBox,
            MBM_WithdrawMoneyCopperBox
        )
        local totalCopper = MBM_MoneyToCopper(g, s, c)

        if totalCopper <= 0 then
            UIErrorsFrame:AddMessage(
                L("Zadej částku k výběru.", "Enter amount to withdraw."),
                1, 0.2, 0.2
            )
            return
        end

        -- 1) clamp podle zůstatku v úschově
        local stored = MBM_GetStoredMoneyCopper()
        if totalCopper > stored then
            totalCopper = stored
        end

        -- 2) clamp podle money capu v inventáři
        local curMoney = GetMoney() or 0
        local canAdd = 2147483640 - curMoney
        if canAdd < 0 then
            canAdd = 0
        end

        if canAdd <= 0 then
            UIErrorsFrame:AddMessage(
                L("Máš u sebe maximum peněz, nelze vybrat další.", "You already have the maximum money, you can't withdraw more."),
                1, 0.2, 0.2
            )
            return
        end

        if totalCopper > canAdd then
            totalCopper = canAdd
        end

        if totalCopper <= 0 then
            UIErrorsFrame:AddMessage(
                L("Nelze vybrat žádné peníze (limit inventáře / prázdná úschova).", "Cannot withdraw any money (inventory cap / empty storage)."),
                1, 0.2, 0.2
            )
            return
        end

        do
            local gg = math.floor(totalCopper / 10000)
            local ss = math.floor((totalCopper % 10000) / 100)
            local cc = totalCopper % 100

            if MBM_WithdrawMoneyGoldBox then MBM_WithdrawMoneyGoldBox:SetText(tostring(gg)) end
            if MBM_WithdrawMoneySilverBox then MBM_WithdrawMoneySilverBox:SetText(tostring(ss)) end
            if MBM_WithdrawMoneyCopperBox then MBM_WithdrawMoneyCopperBox:SetText(tostring(cc)) end
        end

        -- PŘÍKAZ: VYBRAT PENÍZE
        SendChatMessage(string.format(".mb money withdraw %d", totalCopper), "SAY")

		MaterialBankCache = MaterialBankCache or {}
		MaterialBankCache._money = MaterialBankCache._money or {}
		local key = UnitFactionGroup("player") or "Neutral"
		MaterialBankCache._money[key] = MaterialBankCache._money[key] or { copper = 0 }
		local sv = MaterialBankCache._money[key]
		
		sv._pendingWithdraw = totalCopper
		sv._pendingWithdrawTs = GetTime() or 0
		
		-- OKAMŽITÝ odečet pro "Uskladněno:"
		sv.copper = math.max(0, (sv.copper or 0) - totalCopper)
		
		MBM_UpdateTopStripDepositControls()
    end)


    MBM_StoredMoneyText:Hide()
    MBM_WithdrawMoneyGoldBox:Hide()
    MBM_WithdrawMoneySilverBox:Hide()
    MBM_WithdrawMoneyCopperBox:Hide()
    MBM_WithdrawMoneyButton:Hide()
    if MBM_WithdrawMoneyGoldLabel then MBM_WithdrawMoneyGoldLabel:Hide() end
    if MBM_WithdrawMoneySilverLabel then MBM_WithdrawMoneySilverLabel:Hide() end
    if MBM_WithdrawMoneyCopperLabel then MBM_WithdrawMoneyCopperLabel:Hide() end
end

-------------------------------------------------
-- HARD RELOAD IMPLEMENTACE
-------------------------------------------------
local MBM_HardReloadFrame = CreateFrame("Frame")
MBM_HardReloadFrame:Hide()

local MBM_HardReloadSchedule = {0.05, 0.10}

local function MBM_HardReloadNow()
    if not MBM_MainFrame or not MBM_MainFrame:IsShown() then
        return
    end
    MBM_ApplyLocaleStaticTexts()

    local catScroll = 0
    if MBM_CategoryScrollFrame and MBM_CategoryScrollFrame.GetVerticalScroll then
        catScroll = MBM_CategoryScrollFrame:GetVerticalScroll() or 0
    end

    local itemScroll = 0
    if MBM_ItemScrollFrame and MBM_ItemScrollFrame.GetVerticalScroll then
        itemScroll = MBM_ItemScrollFrame:GetVerticalScroll() or 0
    end

    MBM_UpdateModeButtons()
    MBM_UpdateTopStripDepositControls()

    MBM_BuildCategoryList()
    MBM_RebuildItemList()
    MBM_UpdateTopStripDepositControls()

    if MBM_CategoryScrollFrame and MBM_CategoryScrollFrame.SetVerticalScroll then
        MBM_CategoryScrollFrame:SetVerticalScroll(catScroll)
    end
    if MBM_ItemScrollFrame and MBM_ItemScrollFrame.SetVerticalScroll then
        MBM_ItemScrollFrame:SetVerticalScroll(itemScroll)
    end

    if MBM_CategoryScrollFrame then
        MBM_UpdateScrollState(MBM_CategoryScrollFrame)
    end
    if MBM_ItemScrollFrame then
        MBM_UpdateScrollState(MBM_ItemScrollFrame)
    end
end

MBM_RequestHardReload = function()
    if not MBM_MainFrame or not MBM_MainFrame:IsShown() then
        return
    end

    MBM_HardReloadFrame._t = 0
    MBM_HardReloadFrame._i = 1
    MBM_HardReloadFrame:Show()
end

MBM_HardReloadFrame:SetScript(
    "OnUpdate",
    function(self, elapsed)
        self._t = (self._t or 0) + (elapsed or 0)

        local i = self._i or 1
        local target = MBM_HardReloadSchedule[i]
        if not target then
            self:Hide()
            return
        end

        if self._t >= target then
            self._i = i + 1
            MBM_HardReloadNow()
        end
    end
)

local function MBM_NormalizeChatMsg(msg)
    if not msg then
        return ""
    end

    msg = msg:gsub("|c%x%x%x%x%x%x%x%x", ""):gsub("|r", ""):gsub("|H.-|h", ""):gsub("|T.-|t", "")

    msg = msg:gsub("^%s+", ""):gsub("%s+$", "")
    return msg
end

local function MBM_IsBankerNoteMessage(msg)
    local m = MBM_NormalizeChatMsg(msg)
    if m == "" then
        return false
    end

    local CS_PREFIX1 = "Předmět byl automaticky zařazen do kategorie"
    local EN_PREFIX1 = "Item was automatically sorted into category"

    local CS_PREFIX2 = "Poznámka: Předmět byl uložen do kategorie"
    local EN_PREFIX2 = "Note: This item was stored in category"

    if string.find(m, CS_PREFIX1, 1, true) then
        return true
    end
    if string.find(m, EN_PREFIX1, 1, true) then
        return true
    end
    if string.find(m, CS_PREFIX2, 1, true) then
        return true
    end
    if string.find(m, EN_PREFIX2, 1, true) then
        return true
    end

    return false
end

-------------------------------------------------
-- SYNC parser: money
-------------------------------------------------
local function MBM_ParseMoneyToCopperFromText(text)
    if not text or text == "" then
        return nil
    end

    local m = MBM_NormalizeChatMsg(text)
    if m == "" then
        return nil
    end

    if m:find("SYNC", 1, true) then
        return nil
    end

    local low = string.lower(m)
    if not (
        low:find("usk", 1, true) or
        low:find("stored", 1, true) or
        low:find("bank", 1, true) or
        low:find("banka", 1, true) or
        low:find("money", 1, true) or
        low:find("pen", 1, true)
    ) then
        return nil
    end

    local g = m:match("([%d,]+)%s*[gG]")
    local s = m:match("(%d+)%s*[sS]")
    local c = m:match("(%d+)%s*[cC]")

    if g or s or c then
        g = tonumber((select(1, (g or "0"):gsub(",", "")))) or 0
        s = tonumber(s) or 0
        c = tonumber(c) or 0
        return (g * 10000) + (s * 100) + c
    end

    return nil
end

local function MBM_TryHandleMoneySyncMessage(msg)
    local copper = MBM_ParseMoneyToCopperFromText(msg)
    if copper == nil then
        return false
    end

    MBM_SetStoredMoneyFromServer(copper)
    return true
end

local function MBM_ChatMsgFilter(self, event, msg, ...)
    if MBM_OpenFromMinimap and MBM_MainFrame and MBM_MainFrame:IsShown() then
        if MBM_IsBankerNoteMessage(msg) then
            return true
        end
    end
    return false, msg, ...
end

-------------------------------------------------
-- Event handler
-------------------------------------------------
local ev = CreateFrame("Frame")
ev:RegisterEvent("ADDON_LOADED")
ev:RegisterEvent("PLAYER_LOGIN")
ev:RegisterEvent("GET_ITEM_INFO_RECEIVED")
ev:RegisterEvent("BAG_UPDATE")
ev:RegisterEvent("BAG_UPDATE_DELAYED")
ev:RegisterEvent("CHAT_MSG_SYSTEM")
ev:RegisterEvent("PLAYER_MONEY")


ev:SetScript(
    "OnEvent",
    function(self, event, ...)
        if event == "ADDON_LOADED" then
            local name = ...
            if name ~= addonName then
                return
            end

            MBM_SelectedMode = MBM_GetModeSV() or "withdraw"
        elseif event == "PLAYER_LOGIN" then
            MBM_CreateMinimapButton()

            if not MBM_ChatFilterInstalled then
                MBM_ChatFilterInstalled = true

                local events = {
                    "CHAT_MSG_SYSTEM",
                    "CHAT_MSG_SAY",
                    "CHAT_MSG_MONSTER_SAY",
                    "CHAT_MSG_MONSTER_YELL",
                    "CHAT_MSG_MONSTER_WHISPER",
                    "CHAT_MSG_WHISPER",
                    "CHAT_MSG_RAID_BOSS_WHISPER",
                    "CHAT_MSG_RAID_BOSS_EMOTE"
                }

                for _, e in ipairs(events) do
                    ChatFrame_AddMessageEventFilter(e, MBM_ChatMsgFilter)
                end
            end

            if not MBM_WorldHooked and WorldFrame and WorldFrame.HookScript then
                MBM_WorldHooked = true
                WorldFrame:HookScript(
                    "OnMouseDown",
                    function()
                        MBM_ClearAnyEditFocus()
                    end
                )
            end
        elseif event == "GET_ITEM_INFO_RECEIVED" then
            local itemId, success = ...
            if success and MBM_ItemInfoPending[itemId] then
                MBM_ItemInfoPending[itemId] = nil
                if MBM_MainFrame and MBM_MainFrame:IsShown() then
                    MBM_RebuildItemList()
                end
            end

        elseif event == "CHAT_MSG_SYSTEM" then
            local msg = ...

            local m = MBM_NormalizeChatMsg(msg)

            -- =========================================================
            -- COOLDOWN GUARD
            -- =========================================================
            local cd =
                m:match("SYNC%s+cooldown=(%d+)") or
                m:match("SYNC%s+cd=(%d+)") or
                m:match("SYNC%s+wait=(%d+)") or
                m:match("SYNC%s+next=(%d+)")

            if cd then
                MaterialBankCache = MaterialBankCache or {}
                MaterialBankCache._syncCooldownUntil = (GetTime() or 0) + (tonumber(cd) or 0)
                return
            end

            local copperStr = m:match("SYNC%s+money=(%d+)")
            if copperStr then
                MBM_SetStoredMoneyFromServer(tonumber(copperStr) or 0)
                return
            end



            -- =========================================================
            -- MONEY WITHDRAW CONFIRM (aby SV nespadlo na ořezu / capu)
            -- =========================================================
            do
                local money = MaterialBankCache and MaterialBankCache._money
                if money then
                    local key = UnitFactionGroup("player") or "Neutral"
                    local sv = money[key]
                    local pending = sv and sv._pendingWithdraw

                    if pending and pending > 0 then
						if string.find(m, "Vybráno", 1, true) or string.find(m, "Vybrano", 1, true) or
						string.find(m, "Withdrawn", 1, true) then
						
							local g = m:match("([%d,]+)%s*[gG]")
							local s = m:match("(%d+)%s*[sS]")
							local c = m:match("(%d+)%s*[cC]")
						
							local amt = nil
							if g or s or c then
								local gg = tonumber((select(1, (g or "0"):gsub(",", "")))) or 0
								local ss = tonumber(s) or 0
								local cc = tonumber(c) or 0
								amt = (gg * 10000) + (ss * 100) + cc
							end
						
							if not amt or amt <= 0 then
								amt = pending
							end
						
							sv.copper = math.max(0, (sv.copper or 0) + (pending - amt))
						
							sv._pendingWithdraw = nil
							sv._pendingWithdrawTs = nil
						
							if MBM_MainFrame and MBM_MainFrame:IsShown() then
								MBM_UpdateTopStripDepositControls()
							end
							return
						end

						if string.find(m, "nelze vybrat", 1, true) or
						string.find(m, "maximum peněz", 1, true) or
						string.find(m, "Cannot withdraw", 1, true) or
						string.find(m, "maximum money", 1, true) or
						string.find(m, "don't have any stored", 1, true) or
						string.find(m, "nemáš žádné peníze", 1, true) then
						
							sv.copper = math.max(0, (sv.copper or 0) + pending)
						
							sv._pendingWithdraw = nil
							sv._pendingWithdrawTs = nil
						
							if MBM_MainFrame and MBM_MainFrame:IsShown() then
								MBM_UpdateTopStripDepositControls()
							end
							return
						end
                    end
                end
            end

        elseif event == "BAG_UPDATE" or event == "BAG_UPDATE_DELAYED" then
            MBM_RequestDepositRefresh()
			
		elseif event == "PLAYER_MONEY" then
			if MBM_MainFrame and MBM_MainFrame:IsShown() then
				MBM_UpdateTopStripDepositControls()
			end
        end
    end
)
