local ADDON_PREFIX = "Transmog"

-- =========================
-- SavedVariables
-- =========================
local function TMOG_GetMinimapSV()
  TMOG_Saved = TMOG_Saved or { chars = {} }
  TMOG_Saved._minimap = TMOG_Saved._minimap or {}
  local sv = TMOG_Saved._minimap
  if not sv.angle then
    sv.angle = 210
  end
  return sv
end

local CharData = nil
local ActivePreset = 1

local SLOT_ORDER = {
  "HEAD", "SHOULDER", "BACK", "CHEST", "WRIST",
  "HANDS", "WAIST", "LEGS", "FEET", "SHIRT", "TABARD",
  "MAINHAND", "OFFHAND", "RANGED",
}

local function GetCharKey()
  local realm = GetRealmName() or "UnknownRealm"
  local name = UnitName("player") or "UnknownPlayer"
  return realm .. "-" .. name
end

local function EnsureCharData()
  local key = GetCharKey()
  TMOG_Saved.chars[key] = TMOG_Saved.chars[key] or { activePreset = 1, presets = {} }

  local cd = TMOG_Saved.chars[key]
  cd.presets = cd.presets or {}

  for i = 1, 4 do
    cd.presets[i] = cd.presets[i] or {}
  end

  cd.activePreset = tonumber(cd.activePreset or 1) or 1
  if cd.activePreset < 1 then cd.activePreset = 1 end
  if cd.activePreset > 4 then cd.activePreset = 4 end

  CharData = cd
  ActivePreset = cd.activePreset
end

local function GetPresetTable()
  if not CharData then return nil end
  CharData.presets = CharData.presets or {}
  CharData.presets[ActivePreset] = CharData.presets[ActivePreset] or {}
  return CharData.presets[ActivePreset]
end

local function GetEquippedEquipLoc(invSlot)
  local link = GetInventoryItemLink("player", invSlot)
  if not link then return nil end
  local _, _, _, _, _, _, _, _, equipLoc = GetItemInfo(link)
  return equipLoc
end

local function IsDualWieldingTwoHanders()
  local mh = GetEquippedEquipLoc(16)
  local oh = GetEquippedEquipLoc(17)
  return (mh == "INVTYPE_2HWEAPON") and (oh == "INVTYPE_2HWEAPON")
end

-- forward deklarace
local EnsureCache

local PendingLearned = {}

local function EnsureSetForSlot(slotKey)
  local c = EnsureCache(slotKey)
  if not c.set then
    c.set = {}
    for i = 1, #(c.itemIDs or {}) do
      c.set[c.itemIDs[i]] = true
    end
  end
  return c
end

local function InsertSortedUnique(c, itemID)
  itemID = tonumber(itemID or 0) or 0
  if itemID <= 0 then return false end

  c.itemIDs = c.itemIDs or {}
  c.set = c.set or {}

  if c.set[itemID] then
    return false
  end

  local list = c.itemIDs
  local lo, hi = 1, #list
  while lo <= hi do
    local mid = math.floor((lo + hi) / 2)
    local v = list[mid]
    if v == itemID then
      c.set[itemID] = true
      return false
    elseif v < itemID then
      lo = mid + 1
    else
      hi = mid - 1
    end
  end

  table.insert(list, lo, itemID)
  c.set[itemID] = true
  return true
end

-- =========================
-- Cache
-- =========================
local Cache = {}
local SelectedSlotKey = nil

local function Print(msg)
  DEFAULT_CHAT_FRAME:AddMessage("|cff66ccffTransmog:|r " .. tostring(msg))
end

-- =========================
-- Minimap button
-- =========================
local TMOG_MinimapButton
local TMOG_RADIUS = 82

local TMOG_ToggleUI

local function TMOG_UpdateMinimapPosition()
  if not TMOG_MinimapButton or not Minimap then return end

  local sv = TMOG_GetMinimapSV()
  local angle = sv.angle or 210

  local x = math.cos(math.rad(angle)) * TMOG_RADIUS
  local y = math.sin(math.rad(angle)) * TMOG_RADIUS

  TMOG_MinimapButton:ClearAllPoints()
  TMOG_MinimapButton:SetPoint("CENTER", Minimap, "CENTER", x, y)
end

local function TMOG_CreateMinimapButton()
  if TMOG_MinimapButton or not Minimap then return end

  local btn = CreateFrame("Button", "TMOG_MinimapButton", Minimap)
  TMOG_MinimapButton = btn

  btn:SetSize(32, 32)
  btn:SetFrameStrata("LOW")
  btn:SetFrameLevel(8)
  btn:EnableMouse(true)

  local overlay = btn:CreateTexture(nil, "OVERLAY")
  overlay:SetTexture("Interface\\AddOns\\Transmog-Collection\\Textures\\border")
  overlay:SetSize(32, 32)
  overlay:SetPoint("CENTER")

  -- Icon
  local icon = btn:CreateTexture(nil, "ARTWORK")
  icon:SetTexture("Interface\\AddOns\\Transmog-Collection\\Textures\\tmog_icon")
  icon:SetTexCoord(0, 1, 0, 1)
  icon:SetSize(26, 26)
  icon:ClearAllPoints()
  icon:SetPoint("CENTER", btn, "CENTER", 0, -1)
  btn.icon = icon

  btn:SetHighlightTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight")

  btn:RegisterForClicks("LeftButtonUp")
  btn:RegisterForDrag("LeftButton")

  btn:SetScript("OnDragStart", function(self)
    self:SetScript("OnUpdate", function()
      if not Minimap then return end

      local mx, my = Minimap:GetCenter()
      local cx, cy = GetCursorPosition()
      local scale = UIParent:GetEffectiveScale()
      cx, cy = cx / scale, cy / scale

      local angle = math.deg(math.atan2(cy - my, cx - mx))
      if angle < 0 then angle = angle + 360 end

      local sv = TMOG_GetMinimapSV()
      sv.angle = angle

      TMOG_UpdateMinimapPosition()
    end)
  end)

  btn:SetScript("OnDragStop", function(self)
    self:SetScript("OnUpdate", nil)
  end)

  btn:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_LEFT")
    GameTooltip:ClearLines()

    if not GameTooltip._tmogPrevBackdrop then
      local r, g, b, a = GameTooltip:GetBackdropColor()
      local br, bg, bb, ba = GameTooltip:GetBackdropBorderColor()
      GameTooltip._tmogPrevBackdrop = { r, g, b, a, br, bg, bb, ba }
    end

    -- černé pozadí + průhlednost
    GameTooltip:SetBackdropColor(0, 0, 0, 1)
    GameTooltip:SetBackdropBorderColor(1, 1, 1, 1)

    GameTooltip:AddLine("|cffffd100Transmog|r")
    GameTooltip:AddLine("|cff1eff00Left-Click|r |cffffd100Open transmog|r", 1, 1, 1, true)
    GameTooltip:AddLine("|cffccccccLeft-Click + Drag|r |cffffd100Move Minimap Button|r", 1, 1, 1, true)

    GameTooltip:Show()

    local w = GameTooltip:GetWidth() or 0
    local pad = 20

    if w > 0 then
      if GameTooltipTextLeft1 then
        GameTooltipTextLeft1:SetWidth(w - pad)
        GameTooltipTextLeft1:SetJustifyH("CENTER")
      end
      if GameTooltipTextLeft2 then
        GameTooltipTextLeft2:SetWidth(w - pad)
        GameTooltipTextLeft2:SetJustifyH("LEFT")
      end
      if GameTooltipTextLeft3 then
        GameTooltipTextLeft3:SetWidth(w - pad)
        GameTooltipTextLeft3:SetJustifyH("LEFT")
      end
    end
  end)

  btn:SetScript("OnLeave", function()
    GameTooltip:Hide()
    local p = GameTooltip._tmogPrevBackdrop
    if p then
      GameTooltip:SetBackdropColor(p[1], p[2], p[3], p[4])
      GameTooltip:SetBackdropBorderColor(p[5], p[6], p[7], p[8])
      GameTooltip._tmogPrevBackdrop = nil
    end

    if GameTooltipTextLeft1 then
      GameTooltipTextLeft1:SetJustifyH("LEFT")
      GameTooltipTextLeft1:SetWidth(0)
    end
    if GameTooltipTextLeft2 then GameTooltipTextLeft2:SetWidth(0) end
    if GameTooltipTextLeft3 then GameTooltipTextLeft3:SetWidth(0) end
  end)

  btn:SetScript("OnClick", function(_, button)
    if button == "LeftButton" then
      if TMOG_ToggleUI then
        TMOG_ToggleUI()
      end
    end
  end)

  TMOG_UpdateMinimapPosition()
end

EnsureCache = function(slotKey)
  if not Cache[slotKey] then
    Cache[slotKey] = { itemIDs = {}, ready = false, assembling = nil }
  end
  return Cache[slotKey]
end

local function Trim(s)
  s = tostring(s or "")
  s = string.gsub(s, "^%s+", "")
  s = string.gsub(s, "%s+$", "")
  s = string.gsub(s, "[\r\n]", "")
  return s
end

local SlotButtons = {}

local function ClearSelectedSlot()
  SelectedSlotKey = nil
  for _, btn in pairs(SlotButtons) do
    if btn.sel then btn.sel:Hide() end
  end
end

local PendingSlotIcons = {}
local ActiveTransmog = {}

local function SetButtonIcon(btn, texture)
  if not btn or not btn.itemIcon then return end

  if texture then
    btn.itemIcon:SetTexture(texture)
    btn.itemIcon:SetDrawLayer("ARTWORK", 7)
    btn.itemIcon:Show()
  else
    btn.itemIcon:SetTexture(nil)
    btn.itemIcon:Hide()
  end
end

local function UpdateClearButton(slotKey)
  slotKey = Trim(slotKey)
  local btn = SlotButtons[slotKey]
  if not btn or not btn.clearBtn then return end

  if ActiveTransmog[slotKey] then
    btn.clearBtn:Show()
  else
    btn.clearBtn:Hide()
  end
end

-- forward deklarace pro preloader
local QueuePreload

local function ApplySlotIcon(slotKey, itemID)
  slotKey = Trim(slotKey)
  local btn = SlotButtons[slotKey]
  if not btn then return end

  itemID = tonumber(itemID or 0) or 0

  if itemID <= 0 then
    PendingSlotIcons[slotKey] = nil
    ActiveTransmog[slotKey] = nil
    SetButtonIcon(btn, nil)
    UpdateClearButton(slotKey)

    local p = GetPresetTable()
    if p then p[slotKey] = nil end
    return
  end

  local p = GetPresetTable()
  if p then p[slotKey] = itemID end

  local _, _, _, _, _, _, _, _, _, icon = GetItemInfo(itemID)
  if icon then
    SetButtonIcon(btn, icon)
    PendingSlotIcons[slotKey] = nil
    ActiveTransmog[slotKey] = true
    UpdateClearButton(slotKey)
  else
    PendingSlotIcons[slotKey] = itemID
    QueuePreload(itemID)
  end
end

local IconRetry = CreateFrame("Frame")
local _acc = 0
IconRetry:SetScript("OnUpdate", function(_, elapsed)
  _acc = _acc + elapsed
  if _acc < 0.25 then return end
  _acc = 0

  for slotKey, itemID in pairs(PendingSlotIcons) do
    ApplySlotIcon(slotKey, itemID)
  end
end)

-- =========================
-- Client item cache preloader
-- =========================
local PreloadTip = CreateFrame("GameTooltip", "TMOG_PreloadTip", UIParent, "GameTooltipTemplate")
PreloadTip:SetOwner(UIParent, "ANCHOR_NONE")
PreloadTip:Hide()

local PreloadQ = {}
local PreloadSet = {}

QueuePreload = function(itemID)
  itemID = tonumber(itemID or 0) or 0
  if itemID <= 0 then return end
  if PreloadSet[itemID] then return end
  PreloadSet[itemID] = true
  table.insert(PreloadQ, itemID)
end

local function QueuePreloadList(list)
  if not list then return end
  for i = 1, #list do
    QueuePreload(list[i])
  end
end

-- =========================
-- DEBUG: fill cache with many icons
-- =========================
local TMOG_DEBUG_FILL = false
local TMOG_DEBUG_SLOT = "HEAD"
local TMOG_DEBUG_COUNT = 120

local function TMOG_DebugFillSlot()
  if not TMOG_DEBUG_FILL then return end

  local ids = {
    6948,
    117,
    159,
    4540,
    17031,
    1710,
    4604,
    4536,
    4602,
    6265,
  }

  local c = EnsureCache(TMOG_DEBUG_SLOT)
  c.itemIDs = {}
  for i = 1, TMOG_DEBUG_COUNT do
    c.itemIDs[i] = ids[((i - 1) % #ids) + 1]
  end

  c.ready = true
  c.assembling = nil
  c.set = nil

  QueuePreloadList(c.itemIDs)
end

local PreloadFrame = CreateFrame("Frame")
local _preAcc = 0
local PRELOAD_TICK = 0.10
local PRELOAD_PER_TICK = 2

PreloadFrame:SetScript("OnUpdate", function(_, elapsed)
  _preAcc = _preAcc + elapsed
  if _preAcc < PRELOAD_TICK then return end
  _preAcc = 0

  for i = 1, PRELOAD_PER_TICK do
    local id = table.remove(PreloadQ, 1)
    if not id then return end

    if not GetItemInfo(id) then
      PreloadTip:ClearLines()
      PreloadTip:SetHyperlink("item:" .. tostring(id))
      PreloadTip:Hide()
    end
  end
end)

-- =========================
-- SEND
-- =========================
local function Send(msg)
  local target = UnitName("player")
  if SendAddonMessage then
    SendAddonMessage(ADDON_PREFIX, msg, "WHISPER", target)
    return
  end

  local full = ADDON_PREFIX .. "\t" .. msg
  SendChatMessage(full, "WHISPER", nil, target)
end

-- =========================
-- UI
-- =========================
local Main = CreateFrame("Frame", "TMOG_MainFrame", UIParent, "UIPanelDialogTemplate")
Main:SetSize(360, 480)
Main:SetPoint("CENTER")
Main:Hide()
Main:SetFrameStrata("FULLSCREEN_DIALOG")
Main:SetToplevel(true)

-- moveable
Main:SetMovable(true)
Main:EnableMouse(true)
Main:RegisterForDrag("LeftButton")
Main:SetScript("OnDragStart", function(self) self:StartMoving() end)
Main:SetScript("OnDragStop", function(self) self:StopMovingOrSizing() end)

Main.title = Main:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
Main.title:SetPoint("TOP", 0, -8)
Main.title:SetText("Transmog - Preset 1")

local function UpdateTitle()
  Main.title:SetText("Transmog - Preset " .. tostring(ActivePreset or 1))
end

-- Preset buttons (1-4)
local PresetBtns = {}

local function UpdatePresetButtons()
  for i = 1, 4 do
    local b = PresetBtns[i]
    if b then
      if i == ActivePreset then
        b:LockHighlight()
      else
        b:UnlockHighlight()
      end
    end
  end
  UpdateTitle()
end

local function BeginAppliedSync()
  local p = GetPresetTable()
  if p then
    for k in pairs(p) do
      p[k] = nil
    end
  end

  for _, slotKey in ipairs(SLOT_ORDER) do
    PendingSlotIcons[slotKey] = nil
    ActiveTransmog[slotKey] = nil

    local btn = SlotButtons[slotKey]
    if btn then
      SetButtonIcon(btn, nil)
      UpdateClearButton(slotKey)
    end
  end
end

local function LoadPresetToUI(presetIndex)
  presetIndex = tonumber(presetIndex or 1) or 1
  if presetIndex < 1 then presetIndex = 1 end
  if presetIndex > 4 then presetIndex = 4 end

  ActivePreset = presetIndex
  if CharData then
    CharData.activePreset = ActivePreset
  end

  ClearSelectedSlot()
  UpdatePresetButtons()

  BeginAppliedSync()

  Send("SET_PRESET|" .. tostring(ActivePreset))
  Send("GET_APPLIED")
end

do
  local startX = 12
  local y = -35
  local w = 78
  local h = 18
  local gap = 6

  for i = 1, 4 do
    local b = CreateFrame("Button", nil, Main, "UIPanelButtonTemplate")
    b:SetSize(w, h)
    b:SetPoint("TOPLEFT", Main, "TOPLEFT", startX + (i - 1) * (w + gap), y)
    b:SetText("Preset " .. tostring(i))
    b:SetScript("OnClick", function()
      LoadPresetToUI(i)
    end)
    PresetBtns[i] = b
  end
end

-- Model
local Model = CreateFrame("DressUpModel", "TMOG_ModelFrame", Main)
Model:SetSize(245, 330)
Model:SetPoint("TOP", 0, -58)
Model:SetBackdrop({
  bgFile = "Interface\\ChatFrame\\ChatFrameBackground",
  edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
  tile = true, tileSize = 16, edgeSize = 16,
  insets = { left = 4, right = 4, top = 4, bottom = 4 }
})
Model:SetBackdropColor(0, 0, 0, 0.35)

local ModelRotation = 0

local function ResetModelPreview()
  if Model and Model.SetUnit then
    Model:SetUnit("player")
    if Model.SetRotation then
      Model:SetRotation(ModelRotation or 0)
    end
  end
end

local ResetPreviewBtn = CreateFrame("Button", nil, Main, "UIPanelButtonTemplate")
ResetPreviewBtn:SetSize(110, 20)
ResetPreviewBtn:SetPoint("TOPLEFT", Main, "TOPLEFT", 10, -388)
ResetPreviewBtn:SetText("Reset transmog")
ResetPreviewBtn:SetScript("OnClick", function()
  for _, slotKey in ipairs(SLOT_ORDER) do
    Send("CLEAR|" .. slotKey .. "|" .. tostring(ActivePreset or 1))
  end
end)

local ApplyTransmogBtn = CreateFrame("Button", nil, Main, "UIPanelButtonTemplate")
ApplyTransmogBtn:SetSize(110, 20)
ApplyTransmogBtn:SetPoint("TOP", ResetPreviewBtn, "BOTTOM", 0, -4)
ApplyTransmogBtn:SetText("Apply transmog")
ApplyTransmogBtn:SetScript("OnClick", function()
  local p = GetPresetTable()
  if not p then return end

  for _, slotKey in ipairs(SLOT_ORDER) do
    local itemID = tonumber(p[slotKey] or 0) or 0
    if itemID > 0 then
      Send("APPLY|" .. slotKey .. "|" .. tostring(itemID) .. "|" .. tostring(ActivePreset or 1))
    end
  end
end)

Model:EnableMouse(true)
Model:SetScript("OnMouseDown", function(self, button)
  if button ~= "LeftButton" then return end
  self._dragging = true
  self._lastX = GetCursorPosition()
end)

Model:SetScript("OnMouseUp", function(self, button)
  if button ~= "LeftButton" then return end
  self._dragging = false
end)

Model:SetScript("OnUpdate", function(self)
  if not self._dragging or not Model.SetRotation then return end
  local x = GetCursorPosition()
  local dx = x - (self._lastX or x)
  self._lastX = x

  ModelRotation = ModelRotation + (dx * 0.015)
  self:SetRotation(ModelRotation)
end)

local Col = CreateFrame("Frame", "TMOG_CollectionFrame", Main, "UIPanelDialogTemplate")
Col:SetSize(350, 480)
Col:SetPoint("LEFT", Main, "RIGHT", -8, 0)
Col:Hide()
Col:SetFrameStrata("FULLSCREEN_DIALOG")
Col:SetToplevel(true)

Col:HookScript("OnHide", function()
  ClearSelectedSlot()
end)

Col.title = Col:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
Col.title:SetPoint("TOP", 0, -8)
Col.title:SetText("Collection")

local ColInfo = Col:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
ColInfo:SetPoint("TOPLEFT", 12, -34)
ColInfo:SetText("")

local RefreshBtn = CreateFrame("Button", nil, Col, "UIPanelButtonTemplate")
RefreshBtn:SetSize(90, 22)
RefreshBtn:SetPoint("TOPRIGHT", -12, -30)
RefreshBtn:SetText("Synchronize")

local function AddToUISpecialFrames(name)
  if not UISpecialFrames then return end
  for i = 1, #UISpecialFrames do
    if UISpecialFrames[i] == name then return end
  end
  table.insert(UISpecialFrames, name)
end

AddToUISpecialFrames("TMOG_MainFrame")
AddToUISpecialFrames("TMOG_CollectionFrame")

Main:HookScript("OnHide", function()
  if Col and Col:IsShown() then Col:Hide() end
  ClearSelectedSlot()
end)

-- =========================
-- Grid kolekce (ikony v mřížce + scroll)
-- =========================
local FIXED_COLS_WITH_FILTER = 6
local MAX_COLS_NO_FILTER = 12
local GRID_ICON = 36
local GRID_GAP = 4
local GRID_TOP = 60
local GRID_BOTTOM = 14
local GRID_SCROLL_RIGHT = 32

local FILTER_W = 65
local FILTER_GAP = 3

local BASE_GRID_LEFT = 12
local FILTER_GRID_LEFT = 12 + FILTER_W + FILTER_GAP
local CUR_GRID_LEFT = FILTER_GRID_LEFT

local GridScroll = CreateFrame("ScrollFrame", "TMOG_GridScroll", Col, "FauxScrollFrameTemplate")
GridScroll:SetPoint("TOPLEFT", CUR_GRID_LEFT, -GRID_TOP)
GridScroll:SetPoint("BOTTOMRIGHT", -12, GRID_BOTTOM)
GridScroll:EnableMouseWheel(true)

local function ForceScrollBarVisibleAndOnTop()
  local sb = _G[GridScroll:GetName() .. "ScrollBar"]
  if not sb then return end

  if sb:GetParent() ~= GridScroll then
    sb:SetParent(GridScroll)
  end

  sb:ClearAllPoints()
  sb:SetPoint("TOPLEFT", GridScroll, "TOPRIGHT", -18, -16)
  sb:SetPoint("BOTTOMLEFT", GridScroll, "BOTTOMRIGHT", -18, 25)

  sb:SetFrameStrata(Col:GetFrameStrata())
  sb:SetFrameLevel(Col:GetFrameLevel() + 50)
  sb:Show()

  local up = _G[sb:GetName() .. "ScrollUpButton"] or sb.ScrollUpButton
  local down = _G[sb:GetName() .. "ScrollDownButton"] or sb.ScrollDownButton
  if up then
    up:SetFrameStrata(sb:GetFrameStrata())
    up:SetFrameLevel(sb:GetFrameLevel() + 1)
    up:Show()
  end
  if down then
    down:SetFrameStrata(sb:GetFrameStrata())
    down:SetFrameLevel(sb:GetFrameLevel() + 1)
    down:Show()
  end
end

ForceScrollBarVisibleAndOnTop()

local FilterFrame = CreateFrame("Frame", nil, Col)
FilterFrame:SetPoint("TOPLEFT", 12, -GRID_TOP)
FilterFrame:SetPoint("BOTTOMLEFT", 12, GRID_BOTTOM)
FilterFrame:SetWidth(FILTER_W)
FilterFrame:Hide()

local WeaponFilterFrame = CreateFrame("Frame", nil, Col)
WeaponFilterFrame:SetPoint("TOPLEFT", 12, -GRID_TOP)
WeaponFilterFrame:SetPoint("BOTTOMLEFT", 12, GRID_BOTTOM)
WeaponFilterFrame:SetWidth(FILTER_W)
WeaponFilterFrame:Hide()

local WeaponFilter = "ALL"
local WeaponFilterSet = nil
local WeaponFilterBtns = {}
local UpdateGrid
local DetermineWeaponFilterSet

local WEAPON_SETS = {
  WEAPON_MIX = {
    { "ALL",      "All" },
    { "AXE_1H",   "1h Axe" },
    { "MACE_1H",  "1h Mace" },
    { "SWORD_1H", "1h Sword" },
    { "DAGGER",   "Dagger" },
    { "FIST",     "Fist" },
    { "AXE_2H",   "2h Axe" },
    { "MACE_2H",  "2h Mace" },
    { "SWORD_2H", "2h Sword" },
    { "POLEARM",  "Polearm" },
    { "STAFF",    "Staff" },
    { "OTHER",    "Other" },
  },

  RANGED = {
    { "ALL", "All" },
    { "BOW", "Bow" },
    { "GUN", "Gun" },
    { "CROSSBOW", "Crossbow" },
    { "WAND", "Wand" },
    { "THROWN", "Thrown" },
  },

  OFFHAND_SHIELD = {
    { "ALL", "All" },
    { "SHIELD", "Shield" },
    { "OTHER", "Other" },
  },
}

local function EnsureWeaponBtn(i)
  if WeaponFilterBtns[i] then return WeaponFilterBtns[i] end

  local b = CreateFrame("Button", nil, WeaponFilterFrame, "UIPanelButtonTemplate")
  b:SetSize(FILTER_W, 18)
  b:SetScript("OnClick", function(self)
    WeaponFilter = self.filterKey or "ALL"

    FauxScrollFrame_SetOffset(GridScroll, 0)
    local sb = getglobal(GridScroll:GetName() .. "ScrollBar")
    if sb then sb:SetValue(0) end

    for _, bb in pairs(WeaponFilterBtns) do
      if bb.filterKey == WeaponFilter then
        bb:LockHighlight()
      else
        bb:UnlockHighlight()
      end
    end

    if Col:IsShown() and SelectedSlotKey then
      UpdateGrid()
    end
  end)

  WeaponFilterBtns[i] = b
  return b
end

local function SetupWeaponFilterButtons(setKey)
  local opts = WEAPON_SETS[setKey or "ONEHAND"] or WEAPON_SETS.ONEHAND

  for i = 1, 12 do
    if WeaponFilterBtns[i] then WeaponFilterBtns[i]:Hide() end
  end

  local valid = false
  for i = 1, #opts do
    if opts[i][1] == WeaponFilter then
      valid = true
      break
    end
  end
  if not valid then WeaponFilter = "ALL" end

  for i = 1, #opts do
    local key, text = opts[i][1], opts[i][2]
    local b = EnsureWeaponBtn(i)
    b.filterKey = key
    b:SetText(text)
    b:ClearAllPoints()
    b:SetPoint("TOPLEFT", 0, -(i - 1) * 20)
    if key == WeaponFilter then b:LockHighlight() else b:UnlockHighlight() end
    b:Show()
  end
end

local function RefreshWeaponFilterUI(slotKey)
  if not DetermineWeaponFilterSet then
    WeaponFilterFrame:Hide()
    return
  end

  WeaponFilterSet = DetermineWeaponFilterSet(slotKey)
  if not WeaponFilterSet then
    WeaponFilterFrame:Hide()
    return
  end

  WeaponFilterFrame:Show()
  SetupWeaponFilterButtons(WeaponFilterSet)
end

local ArmorFilter = "ALL"
local FilterBtns = {}

local function UpdateFilterButtons()
  for k, b in pairs(FilterBtns) do
    if k == ArmorFilter then
      b:LockHighlight()
    else
      b:UnlockHighlight()
    end
  end
end

local function SetArmorFilter(key)
  ArmorFilter = key or "ALL"
  UpdateFilterButtons()
  if Col:IsShown() and SelectedSlotKey then
    UpdateGrid()
  end
end

local function CreateFilterBtn(key, text, yOff)
  local b = CreateFrame("Button", nil, FilterFrame, "UIPanelButtonTemplate")
  b:SetSize(FILTER_W, 18)
  b:SetPoint("TOPLEFT", 0, yOff)
  b:SetText(text)
  b:SetScript("OnClick", function() SetArmorFilter(key) end)
  FilterBtns[key] = b
  return b
end

CreateFilterBtn("ALL", "All", 0)
CreateFilterBtn("CLOTH", "Cloth", -20)
CreateFilterBtn("LEATHER", "Leather", -40)
CreateFilterBtn("MAIL", "Mail", -60)
CreateFilterBtn("PLATE", "Plate", -80)

UpdateFilterButtons()

local GridHolder = CreateFrame("Frame", nil, Col)
GridHolder:SetPoint("TOPLEFT", CUR_GRID_LEFT, -GRID_TOP)
GridHolder:SetPoint("BOTTOMRIGHT", -GRID_SCROLL_RIGHT, GRID_BOTTOM)

local GridButtons = {}

-- =========================
-- Armor filter (HEAD/SHOULDER/CHEST/WRIST/HANDS/WAIST/LEGS/FEET)
-- =========================
local ARMOR_FILTER_SLOTS = {
  HEAD = true, SHOULDER = true, CHEST = true, WRIST = true,
  HANDS = true, WAIST = true, LEGS = true, FEET = true,
}

local function SlotUsesLeftFilter(slotKey)
  return (slotKey == "MAINHAND" or slotKey == "OFFHAND" or slotKey == "RANGED" or ARMOR_FILTER_SLOTS[slotKey] == true)
end

local function GetGridHolderWidth()
  local w = GridHolder and (GridHolder:GetWidth() or 0) or 0
  if w and w > 1 then
    return w
  end

  local cw = Col and (Col:GetWidth() or 350) or 350
  local approx = cw - (CUR_GRID_LEFT or 12) - GRID_SCROLL_RIGHT
  if approx < 1 then approx = 1 end
  return approx
end

local function GetGridColsForSlot(slotKey)
  if SlotUsesLeftFilter(slotKey) then
    return FIXED_COLS_WITH_FILTER
  end

  local w = GetGridHolderWidth()
  local cols = math.floor((w + GRID_GAP) / (GRID_ICON + GRID_GAP))
  if cols < 1 then cols = 1 end
  if cols > MAX_COLS_NO_FILTER then cols = MAX_COLS_NO_FILTER end
  return cols
end

local CUR_GRID_COLS = FIXED_COLS_WITH_FILTER

local function ApplyGridLayoutForSlot(slotKey)
  local useFilter = SlotUsesLeftFilter(slotKey)

  CUR_GRID_LEFT = useFilter and FILTER_GRID_LEFT or BASE_GRID_LEFT

  GridScroll:ClearAllPoints()
  GridScroll:SetPoint("TOPLEFT", Col, "TOPLEFT", CUR_GRID_LEFT, -GRID_TOP)
  GridScroll:SetPoint("BOTTOMRIGHT", Col, "BOTTOMRIGHT", -12, GRID_BOTTOM)

  GridHolder:ClearAllPoints()
  GridHolder:SetPoint("TOPLEFT", Col, "TOPLEFT", CUR_GRID_LEFT, -GRID_TOP)
  GridHolder:SetPoint("BOTTOMRIGHT", Col, "BOTTOMRIGHT", -GRID_SCROLL_RIGHT, GRID_BOTTOM)

  local newCols = GetGridColsForSlot(slotKey)
  if newCols ~= (CUR_GRID_COLS or newCols) then
    CUR_GRID_COLS = newCols

    FauxScrollFrame_SetOffset(GridScroll, 0)
    local sb = getglobal(GridScroll:GetName() .. "ScrollBar")
    if sb then sb:SetValue(0) end
  else
    CUR_GRID_COLS = newCols
  end

  ForceScrollBarVisibleAndOnTop()
end

local function _lc(s) return string.lower(tostring(s or "")) end

local ARMOR_KEYWORDS = {
  CLOTH = { "cloth" },
  LEATHER = { "leather" },
  MAIL = { "mail" },
  PLATE = { "plate" },
}

local function _containsAny(hay, needles)
  for i = 1, #needles do
    if string.find(hay, needles[i], 1, true) then
      return true
    end
  end
  return false
end

local function ItemMatchesArmorFilter(itemID, filterKey)
  local _, _, _, _, _, _, subType = GetItemInfo(itemID)
  if not subType then
    return true
  end

  local st = _lc(subType)
  local keys = ARMOR_KEYWORDS[filterKey]
  if not keys then return true end

  return _containsAny(st, keys)
end

local function ApplyArmorFilter(slotKey, rawItems)
  if not slotKey or not ARMOR_FILTER_SLOTS[slotKey] then
    return rawItems
  end
  if ArmorFilter == "ALL" then
    return rawItems
  end

  local out = {}
  for i = 1, #rawItems do
    local id = rawItems[i]
    if ItemMatchesArmorFilter(id, ArmorFilter) then
      table.insert(out, id)
    end
  end
  return out
end

-- =========================
-- Strict UI filter (MAINHAND/OFFHAND/RANGED) - musí odpovídat server ValidateWeaponStrict
-- =========================
local INV_SLOT = { MAINHAND = 16, OFFHAND = 17, RANGED = 18 }
local StrictCache = {}

local function IsStrictSlot(slotKey)
  return slotKey == "MAINHAND" or slotKey == "OFFHAND" or slotKey == "RANGED"
end

local function GetEquippedEquipLocAndSubType(slotKey)
  local inv = INV_SLOT[slotKey]
  if not inv then return nil, nil end

  local link = GetInventoryItemLink("player", inv)
  if not link then return nil, nil end

  local _, _, _, _, _, _, subType, _, equipLoc = GetItemInfo(link)
  return equipLoc, subType
end

local function IsRangedEquipLoc(loc)
  return loc == "INVTYPE_RANGED"
    or loc == "INVTYPE_RANGEDRIGHT"
    or loc == "INVTYPE_THROWN"
end

local function CanOpenRangedCollection()
  local eqEquipLoc = select(1, GetEquippedEquipLocAndSubType("RANGED"))

  if not eqEquipLoc then
    Print("To transmogrify this slot, you must equip a ranged weapon first.")
    return false
  end

  if eqEquipLoc == "INVTYPE_RELIC" then
    Print("Relics cannot be transmogrified. Please equip a ranged weapon first.")
    return false
  end

  if not IsRangedEquipLoc(eqEquipLoc) then
    Print("To transmogrify this slot, you must equip a ranged weapon first.")
    return false
  end

  return true
end

DetermineWeaponFilterSet = function(slotKey)
  if slotKey == "RANGED" then
    return "RANGED"
  end

  if slotKey ~= "MAINHAND" and slotKey ~= "OFFHAND" then
    return nil
  end

  local eqEquipLoc = select(1, GetEquippedEquipLocAndSubType(slotKey))
  if not eqEquipLoc then
    return nil
  end

  if slotKey == "OFFHAND" and (eqEquipLoc == "INVTYPE_SHIELD" or eqEquipLoc == "INVTYPE_HOLDABLE") then
    return "OFFHAND_SHIELD"
  end

  if eqEquipLoc == "INVTYPE_2HWEAPON"
    or eqEquipLoc == "INVTYPE_WEAPON"
    or eqEquipLoc == "INVTYPE_WEAPONMAINHAND"
    or eqEquipLoc == "INVTYPE_WEAPONOFFHAND" then
    return "WEAPON_MIX"
  end

  return nil
end

local WEAPON_KEYWORDS = {
  WEAPON_MIX = {
    AXE_1H   = { "one-handed axes" },
    MACE_1H  = { "one-handed maces" },
    SWORD_1H = { "one-handed swords" },

    DAGGER   = { "daggers" },
    FIST     = { "fist weapons" },

    AXE_2H   = { "two-handed axes" },
    MACE_2H  = { "two-handed maces" },
    SWORD_2H = { "two-handed swords" },

    POLEARM  = { "polearms" },
    STAFF    = { "staves" },
    OTHER    = { "fishing pole" },
  },

  RANGED = {
    BOW = { "bows" },
    GUN = { "guns" },
    CROSSBOW = { "crossbows" },
    WAND = { "wands" },
    THROWN = { "thrown" },
  },
}

local function ApplyWeaponTypeFilter(slotKey, rawItems)
  if slotKey ~= "MAINHAND" and slotKey ~= "OFFHAND" and slotKey ~= "RANGED" then
    return rawItems
  end

  if not WeaponFilterSet or WeaponFilter == "ALL" then
    return rawItems
  end

  local out = {}

  if WeaponFilterSet == "OFFHAND_SHIELD" then
    for i = 1, #rawItems do
      local id = rawItems[i]
      local _, _, _, _, _, _, _, _, equipLoc = GetItemInfo(id)

      if not equipLoc then
        table.insert(out, id)
      else
        if WeaponFilter == "SHIELD" then
          if equipLoc == "INVTYPE_SHIELD" then table.insert(out, id) end
        elseif WeaponFilter == "OTHER" then
          if equipLoc == "INVTYPE_HOLDABLE" then table.insert(out, id) end
        else
          table.insert(out, id)
        end
      end
    end
    return out
  end

  local map = WEAPON_KEYWORDS[WeaponFilterSet]
  local keys = map and map[WeaponFilter]
  if not keys then
    return rawItems
  end

  for i = 1, #rawItems do
    local id = rawItems[i]
    local _, _, _, _, _, _, subType = GetItemInfo(id)

    if not subType then
      table.insert(out, id)
    else
      local st = _lc(subType)
      if _containsAny(st, keys) then
        table.insert(out, id)
      end
    end
  end

  return out
end

local function IsWeaponEquipLoc_MainhandAllowed(loc)
  return loc == "INVTYPE_2HWEAPON"
    or loc == "INVTYPE_WEAPON"
    or loc == "INVTYPE_WEAPONMAINHAND"
end

local function IsWeaponEquipLoc_OffhandAllowed(loc)
  return loc == "INVTYPE_2HWEAPON"
    or loc == "INVTYPE_WEAPON"
    or loc == "INVTYPE_WEAPONMAINHAND"
    or loc == "INVTYPE_WEAPONOFFHAND"
end

local function IsFistSubType(subType)
  local st = _lc(subType or "")
  return (st ~= "" and (string.find(st, "fist", 1, true) or string.find(st, "pěst", 1, true)))
end

local function ItemMatchesStrict(slotKey, itemID, eqEquipLoc, eqSubType)
  local _, _, _, _, _, _, subType, _, equipLoc = GetItemInfo(itemID)

  if not equipLoc then
    return true
  end

  if not eqEquipLoc then
    if slotKey == "OFFHAND" or slotKey == "RANGED" then
      return false
    end
    return true
  end

  if slotKey == "MAINHAND" then
    if equipLoc == "INVTYPE_WEAPONOFFHAND" and IsFistSubType(subType) then
      return false
    end

    return IsWeaponEquipLoc_MainhandAllowed(equipLoc)
  end

  if slotKey == "OFFHAND" then
    if eqEquipLoc == "INVTYPE_SHIELD" or eqEquipLoc == "INVTYPE_HOLDABLE" then
      return equipLoc == "INVTYPE_SHIELD" or equipLoc == "INVTYPE_HOLDABLE"
    end

    if equipLoc == "INVTYPE_WEAPONMAINHAND" and IsFistSubType(subType) then
      return false
    end

    return IsWeaponEquipLoc_OffhandAllowed(equipLoc)
  end

  if slotKey == "RANGED" then
    local function isRanged(loc)
      return loc == "INVTYPE_RANGED"
        or loc == "INVTYPE_RANGEDRIGHT"
        or loc == "INVTYPE_THROWN"
    end

    if not isRanged(eqEquipLoc) then
      return false
    end

    return isRanged(equipLoc)
  end

  return true
end

local function GetFilteredItemsForSlot(slotKey, rawItems)
  if not IsStrictSlot(slotKey) then
    return rawItems
  end

  local eqEquipLoc, eqSubType = GetEquippedEquipLocAndSubType(slotKey)
  local sig = tostring(eqEquipLoc or "nil") .. "|" .. tostring(eqSubType or "nil") .. "|" .. tostring(#rawItems)

  local c = StrictCache[slotKey]
  if c and c.sig == sig then
    return c.items
  end

  local out = {}
  for i = 1, #rawItems do
    local id = rawItems[i]
    if ItemMatchesStrict(slotKey, id, eqEquipLoc, eqSubType) then
      table.insert(out, id)
    end
  end

  StrictCache[slotKey] = { sig = sig, items = out }
  return out
end

local function WipeStrictCache()
  for k in pairs(StrictCache) do
    StrictCache[k] = nil
  end
end

local function PreviewItem(itemID)
  if not itemID then return end
  if not Model or not Model.TryOn then return end
  ResetModelPreview()
  Model:TryOn("item:" .. tostring(itemID))
end

local function ApplyFromGrid(itemID)
  if not itemID or not SelectedSlotKey then return end
  Send("APPLY|" .. SelectedSlotKey .. "|" .. tostring(itemID) .. "|" .. tostring(ActivePreset or 1))
end

local function EnsureGridButton(i)
  if GridButtons[i] then return GridButtons[i] end

  local b = CreateFrame("Button", nil, GridHolder)
  b:SetSize(GRID_ICON, GRID_ICON)

  b.icon = b:CreateTexture(nil, "ARTWORK")
  b.icon:SetAllPoints(b)
  b.icon:SetTexture("Interface\\Icons\\INV_Misc_QuestionMark")

  b:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")

  b:SetScript("OnClick", function(self)
    if self.itemID then
      ApplyFromGrid(self.itemID)
    end
  end)

  b:SetScript("OnEnter", function(self)
    if self.itemID then
      GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
      GameTooltip:SetHyperlink("item:" .. tostring(self.itemID))
      GameTooltip:Show()
      PreviewItem(self.itemID)
    end
  end)

  b:SetScript("OnLeave", function()
    GameTooltip:Hide()
    ResetModelPreview()
  end)

  GridButtons[i] = b
  return b
end

local function GetVisibleRows()
  local h = GridHolder:GetHeight() or 1
  local rowH = GRID_ICON + GRID_GAP
  local rows = math.floor((h + GRID_GAP) / rowH)
  if rows < 1 then rows = 1 end
  return rows
end

UpdateGrid = function()
  if not SelectedSlotKey then return end

  local c = EnsureCache(SelectedSlotKey)
  local rawItems = c.itemIDs or {}
  local items = ApplyArmorFilter(SelectedSlotKey, rawItems)
  items = GetFilteredItemsForSlot(SelectedSlotKey, items)
  items = ApplyWeaponTypeFilter(SelectedSlotKey, items)

  ColInfo:SetText("Total: " .. tostring(#items))

  local cols = GetGridColsForSlot(SelectedSlotKey)

  local visibleRows = GetVisibleRows()
  local rowH = GRID_ICON + GRID_GAP
  local totalRows = math.ceil(#items / cols)

  FauxScrollFrame_Update(GridScroll, totalRows, visibleRows, rowH)
  ForceScrollBarVisibleAndOnTop()

  local sb = _G[GridScroll:GetName() .. "ScrollBar"]
  if sb then
    if totalRows <= visibleRows then
      FauxScrollFrame_SetOffset(GridScroll, 0)
      sb:SetValue(0)
      sb:Hide()
      if sb.ScrollUpButton then sb.ScrollUpButton:Hide() end
      if sb.ScrollDownButton then sb.ScrollDownButton:Hide() end
    else
      sb:Show()
      if sb.ScrollUpButton then sb.ScrollUpButton:Show() end
      if sb.ScrollDownButton then sb.ScrollDownButton:Show() end
    end
  end

  local offsetRows = FauxScrollFrame_GetOffset(GridScroll) or 0
  local startIndex = offsetRows * cols + 1
  local visibleCount = visibleRows * cols

  local startX = 0
  if SlotUsesLeftFilter(SelectedSlotKey) then
    local holderW = GridHolder:GetWidth() or 0
    local gridW = (cols * GRID_ICON) + ((cols - 1) * GRID_GAP)
    startX = holderW - gridW
    startX = startX - 5
    if startX < 0 then startX = 0 end
  else
    startX = 0
    startX = startX + 15
  end

  for i = 1, visibleCount do
    local idx = startIndex + (i - 1)
    local b = EnsureGridButton(i)

    if idx <= #items then
      local itemID = items[idx]
      b.itemID = itemID

      local _, _, _, _, _, _, _, _, _, icon = GetItemInfo(itemID)
      b.icon:SetTexture(icon or "Interface\\Icons\\INV_Misc_QuestionMark")

      local col = (i - 1) % cols
      local row = math.floor((i - 1) / cols)

      b:ClearAllPoints()
      b:SetPoint(
        "TOPLEFT",
        GridHolder,
        "TOPLEFT",
        startX + col * (GRID_ICON + GRID_GAP),
        -row * (GRID_ICON + GRID_GAP)
      )
      b:Show()
    else
      b.itemID = nil
      b:Hide()
    end
  end

  for i = visibleCount + 1, #GridButtons do
    if GridButtons[i] then
      GridButtons[i].itemID = nil
      GridButtons[i]:Hide()
    end
  end
end

GridScroll:SetScript("OnVerticalScroll", function(self, offset)
  FauxScrollFrame_OnVerticalScroll(self, offset, (GRID_ICON + GRID_GAP), UpdateGrid)
end)

GridScroll:SetScript("OnMouseWheel", function(self, delta)
  local c = SelectedSlotKey and EnsureCache(SelectedSlotKey) or nil
  local rawItems = (c and c.itemIDs) or {}
  local items = ApplyArmorFilter(SelectedSlotKey, rawItems)
  items = GetFilteredItemsForSlot(SelectedSlotKey, items)
  items = ApplyWeaponTypeFilter(SelectedSlotKey, items)

  local cols = GetGridColsForSlot(SelectedSlotKey)
  local visibleRows = GetVisibleRows()
  local totalRows = math.ceil(#items / cols)

  local rowH = (GRID_ICON + GRID_GAP)
  local offsetRows = FauxScrollFrame_GetOffset(self) or 0

  if delta < 0 then offsetRows = offsetRows + 1 else offsetRows = offsetRows - 1 end
  if offsetRows < 0 then offsetRows = 0 end

  local maxOffset = totalRows - visibleRows
  if maxOffset < 0 then maxOffset = 0 end
  if offsetRows > maxOffset then offsetRows = maxOffset end

  local sb = getglobal(self:GetName() .. "ScrollBar")
  if sb then
    sb:SetValue(offsetRows * rowH)
  else
    FauxScrollFrame_SetOffset(self, offsetRows)
    UpdateGrid()
  end
end)

RefreshBtn:SetScript("OnClick", function()
  if not SelectedSlotKey then return end

  if SelectedSlotKey == "RANGED" then
    if not CanOpenRangedCollection() then
      return
    end
  end

  StrictCache[SelectedSlotKey] = nil
  local c = EnsureCache(SelectedSlotKey)
  c.itemIDs = {}
  c.ready = false
  c.assembling = nil
  ColInfo:SetText("Requesting data from server...")
  Send("REQ|" .. SelectedSlotKey)
end)

-- =========================
-- Slot buttons
-- =========================
local SLOT_TEX = {
  HEAD     = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Head",
  SHOULDER = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Shoulder",
  BACK     = "Interface\\AddOns\\Transmog-Collection\\Textures\\Ui-paperdoll-slot-back",
  CHEST    = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Chest",
  WRIST    = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Wrists",
  HANDS    = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Hands",
  WAIST    = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Waist",
  LEGS     = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Legs",
  FEET     = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Feet",
  SHIRT    = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Shirt",
  TABARD   = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Tabard",
  MAINHAND = "Interface\\PaperDoll\\UI-PaperDoll-Slot-MainHand",
  OFFHAND  = "Interface\\PaperDoll\\UI-PaperDoll-Slot-SecondaryHand",
  RANGED   = "Interface\\PaperDoll\\UI-PaperDoll-Slot-Ranged",
}

local SLOT_NAME = {
  HEAD = "Head", SHOULDER = "Shoulder", BACK = "Back", CHEST = "Chest", WRIST = "Wrist",
  HANDS = "Hands", WAIST = "Waist", LEGS = "Legs", FEET = "Feet",
  SHIRT = "Shirt", TABARD = "Tabard",
  MAINHAND = "Main Hand", OFFHAND = "Off Hand", RANGED = "Ranged",
}

local SLOT_INV = {
  HEAD = 1,
  SHOULDER = 3,
  SHIRT = 4,
  CHEST = 5,
  WAIST = 6,
  LEGS = 7,
  FEET = 8,
  WRIST = 9,
  HANDS = 10,
  BACK = 15,
  MAINHAND = 16,
  OFFHAND = 17,
  RANGED = 18,
  TABARD = 19,
}

local function CanOpenCollectionForSlot(slotKey)
  -- RANGED má speciální pravidla (nic/relic/neranged)
  if slotKey == "RANGED" then
    return CanOpenRangedCollection()
  end

  local inv = SLOT_INV[slotKey]
  if not inv then
    return true
  end

  local link = GetInventoryItemLink("player", inv)
  if not link then
    Print("To transmogrify " .. tostring(SLOT_NAME[slotKey] or slotKey) .. ", you must have an item equipped in that slot.")
    return false
  end

  return true
end

local function SetSelected(button, selected)
  if selected then
    button.sel:Show()
  else
    button.sel:Hide()
  end
end

local function CreateSlotButton(slotKey, point, rel, x, y)
  local b = CreateFrame("Button", nil, Main)
  b:SetSize(40, 40)
  b:SetPoint(point, rel, point, x, y)
  b.slotKey = slotKey
  SlotButtons[slotKey] = b

  b.bg = b:CreateTexture(nil, "BACKGROUND")
  b.bg:SetAllPoints(b)
  b.bg:SetTexture(SLOT_TEX[slotKey] or "Interface\\Icons\\INV_Misc_QuestionMark")

  b:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square", "ADD")

  b.itemIcon = b:CreateTexture(nil, "ARTWORK")
  b.itemIcon:SetPoint("TOPLEFT", 0, 0)
  b.itemIcon:SetPoint("BOTTOMRIGHT", 0, 0)
  b.itemIcon:SetTexture(nil)
  b.itemIcon:Hide()
  b.itemIcon:SetDrawLayer("ARTWORK", 7)

  b.clearBtn = CreateFrame("Button", nil, b, "UIPanelCloseButton")
  b.clearBtn:SetSize(24, 24)
  b.clearBtn:SetPoint("TOPRIGHT", 6, 6)
  b.clearBtn:Hide()
  b.clearBtn:SetScript("OnClick", function()
    Send("CLEAR|" .. slotKey .. "|" .. tostring(ActivePreset or 1))
  end)

  b.sel = b:CreateTexture(nil, "OVERLAY")
  b.sel:SetAllPoints(b)
  b.sel:SetTexture("Interface\\Buttons\\CheckButtonHilight")
  b.sel:SetBlendMode("ADD")
  b.sel:Hide()

  b:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
    GameTooltip:SetText(SLOT_NAME[slotKey] or slotKey)
    GameTooltip:Show()
  end)

  b:SetScript("OnLeave", function()
    GameTooltip:Hide()
  end)

  b:SetScript("OnClick", function(self)
    local sk = self.slotKey

    -- BLOCK: neotevírat kolekci, pokud není nic nasazeno (RANGED má extra pravidla)
    if not CanOpenCollectionForSlot(sk) then
      return
    end

    SelectedSlotKey = sk

    for _, btn in pairs(SlotButtons) do
      SetSelected(btn, btn == self)
    end

    Col:Show()
    Col.title:SetText("Collection: " .. (SLOT_NAME[sk] or sk))

    if ARMOR_FILTER_SLOTS[sk] then
      FilterFrame:Show()
      WeaponFilterFrame:Hide()
    elseif sk == "MAINHAND" or sk == "OFFHAND" or sk == "RANGED" then
      FilterFrame:Hide()
      RefreshWeaponFilterUI(sk)
    else
      FilterFrame:Hide()
      WeaponFilterFrame:Hide()
    end

    ApplyGridLayoutForSlot(sk)

    local c = EnsureCache(SelectedSlotKey)
    if c.ready then
      UpdateGrid()
    else
      ColInfo:SetText("Requesting data from server...")
      Send("REQ|" .. SelectedSlotKey)
    end

    ResetModelPreview()
  end)

  UpdateClearButton(slotKey)
  return b
end

local topY = -66
local step = -44
local leftX = 12
local rightX = Main:GetWidth() - 12 - 40

CreateSlotButton("HEAD", "TOPLEFT", Main, leftX, topY + 0 * step)
CreateSlotButton("SHOULDER", "TOPLEFT", Main, leftX, topY + 1 * step)
CreateSlotButton("BACK", "TOPLEFT", Main, leftX, topY + 2 * step)
CreateSlotButton("CHEST", "TOPLEFT", Main, leftX, topY + 3 * step)
CreateSlotButton("WRIST", "TOPLEFT", Main, leftX, topY + 4 * step)

CreateSlotButton("HANDS", "TOPLEFT", Main, rightX, topY + 0 * step)
CreateSlotButton("WAIST", "TOPLEFT", Main, rightX, topY + 1 * step)
CreateSlotButton("LEGS", "TOPLEFT", Main, rightX, topY + 2 * step)
CreateSlotButton("FEET", "TOPLEFT", Main, rightX, topY + 3 * step)
CreateSlotButton("SHIRT", "TOPLEFT", Main, rightX, topY + 4 * step)
CreateSlotButton("TABARD", "TOPLEFT", Main, rightX, topY + 5 * step)

local bottomY = -430
CreateSlotButton("MAINHAND", "TOPLEFT", Main, 118, bottomY)
CreateSlotButton("OFFHAND", "TOPLEFT", Main, 160, bottomY)
CreateSlotButton("RANGED", "TOPLEFT", Main, 202, bottomY)

UpdatePresetButtons()

-- =========================
-- příkaz
-- =========================
TMOG_ToggleUI = function()
  TMOG_CreateMinimapButton()
  if Main:IsShown() then
    Main:Hide()
    Col:Hide()
    ClearSelectedSlot()
  else
    Main:Show()
    ResetModelPreview()
    UpdatePresetButtons()
    Send("SET_PRESET|" .. tostring(ActivePreset or 1))
    Send("GET_APPLIED")
  end
end

SLASH_TMOG1 = "/tmog"
SlashCmdList["TMOG"] = function()
  TMOG_ToggleUI()
end

-- =========================
-- Addon zpráva
-- =========================
local function Split(str, sep)
  local t = {}
  for part in string.gmatch(str, "([^" .. sep .. "]+)") do
    table.insert(t, part)
  end
  return t
end

local function ParseMsg(msg)
  msg = Trim(msg)
  local parts = Split(msg, "|")
  local cmd = Trim(parts[1])

  if cmd == "RSP" then
    local slotKey = Trim(parts[2])
    if slotKey == "" then return end
    local c = EnsureCache(slotKey)

    local frac = parts[3] or "1/1"
    local csv = parts[4] or ""
    local p = Split(frac, "/")
    local total = tonumber(p[2] or "1") or 1

    if not c.assembling then
      c.assembling = { total = total, got = 0, items = {} }
    end

    if csv ~= "" then
      for idStr in string.gmatch(csv, "([^,]+)") do
        local id = tonumber(idStr)
        if id and id > 0 then
          table.insert(c.assembling.items, id)
        end
      end
    end

    c.assembling.got = c.assembling.got + 1

    if slotKey == SelectedSlotKey then
      ColInfo:SetText("Loading... (" .. tostring(c.assembling.got) .. "/" .. tostring(total) .. ")")
    end

    if c.assembling.got >= c.assembling.total then
      c.itemIDs = c.assembling.items
      c.ready = true
      c.assembling = nil
      StrictCache[slotKey] = nil

      c.set = {}
      for i = 1, #c.itemIDs do
        c.set[c.itemIDs[i]] = true
      end

      if PendingLearned[slotKey] then
        for id, _ in pairs(PendingLearned[slotKey]) do
          InsertSortedUnique(c, id)
        end
        PendingLearned[slotKey] = nil
      end

      QueuePreloadList(c.itemIDs)

      if slotKey == SelectedSlotKey then
        UpdateGrid()
      end
    end

  elseif cmd == "PRESET_OK" then
    local preset = tonumber(Trim(parts[2] or "1")) or 1
    if preset < 1 then preset = 1 end
    if preset > 4 then preset = 4 end
  
    ActivePreset = preset
    if CharData then CharData.activePreset = ActivePreset end
    UpdatePresetButtons()
  
    BeginAppliedSync()
    Send("GET_APPLIED")

  elseif cmd == "APPLIED" then
    local slotKey = Trim(parts[2] or "")
    local entry = tonumber(Trim(parts[3] or "0")) or 0
    if slotKey == "" then return end
    ApplySlotIcon(slotKey, entry)
    ResetModelPreview()

  elseif cmd == "APPLIED_DONE" then
    ResetModelPreview()

  elseif cmd == "APPLY_OK" then
    local okSlot = Trim(parts[2])
    local entry = tonumber(Trim(parts[3])) or 0
    if okSlot ~= "" and entry >= 0 then
      ApplySlotIcon(okSlot, entry)
    end
    ResetModelPreview()

  elseif cmd == "APPLY_ERR" then
    local reason = parts[2] or "UNKNOWN"
    Print("Cannot apply: " .. tostring(reason))

  elseif cmd == "LEARN" then
    local slotKey = Trim(parts[2] or "")
    local itemID = tonumber(Trim(parts[3] or "0")) or 0
    if slotKey == "" or itemID <= 0 then return end

    QueuePreload(itemID)
    local c = EnsureCache(slotKey)

    if c.ready then
      EnsureSetForSlot(slotKey)
      local changed = InsertSortedUnique(c, itemID)
      if changed then
        StrictCache[slotKey] = nil

        if Col:IsShown() and SelectedSlotKey == slotKey then
          UpdateGrid()
        end
      end
    else
      PendingLearned[slotKey] = PendingLearned[slotKey] or {}
      PendingLearned[slotKey][itemID] = true
    end

  elseif cmd == "CLEAR_OK" then
    local slotKey = Trim(parts[2] or "")
    if slotKey ~= "" then
      ApplySlotIcon(slotKey, 0)
    else
      for k, _ in pairs(SlotButtons) do
        ApplySlotIcon(k, 0)
      end
    end
    ResetModelPreview()
  end
end

local Evt = CreateFrame("Frame")
Evt:RegisterEvent("PLAYER_LOGIN")
Evt:RegisterEvent("PLAYER_ENTERING_WORLD")
Evt:RegisterEvent("CHAT_MSG_ADDON")
Evt:RegisterEvent("PLAYER_EQUIPMENT_CHANGED")
Evt:RegisterEvent("UNIT_MODEL_CHANGED")

Evt:SetScript("OnEvent", function(_, event, ...)
  if event == "PLAYER_LOGIN" then
    EnsureCharData()
    TMOG_CreateMinimapButton()

    if RegisterAddonMessagePrefix then
      pcall(RegisterAddonMessagePrefix, ADDON_PREFIX)
    end

    UpdatePresetButtons()

	BeginAppliedSync()
    Send("SET_PRESET|" .. tostring(ActivePreset or 1))
    Send("GET_APPLIED")

    TMOG_DebugFillSlot()

  elseif event == "PLAYER_ENTERING_WORLD" then
    if not _G["TMOG_MinimapButton"] then
      TMOG_CreateMinimapButton()
    end

  elseif event == "CHAT_MSG_ADDON" then
    local prefix, msg = ...
    if prefix ~= ADDON_PREFIX then return end
    ParseMsg(msg)

  elseif event == "PLAYER_EQUIPMENT_CHANGED" then
    if Main:IsShown() then ResetModelPreview() end

    WipeStrictCache()
    if Col:IsShown() and SelectedSlotKey then
      if SelectedSlotKey == "RANGED" then
        if not CanOpenRangedCollection() then
          Col:Hide()
          ClearSelectedSlot()
          return
        end
      end

      if SelectedSlotKey == "MAINHAND" or SelectedSlotKey == "OFFHAND" or SelectedSlotKey == "RANGED" then
        RefreshWeaponFilterUI(SelectedSlotKey)
      end
      UpdateGrid()
    end

  elseif event == "UNIT_MODEL_CHANGED" then
    local unit = ...
    if unit == "player" and Main:IsShown() then
      ResetModelPreview()
    end
  end
end)
