#include "Config.h"
#include "Creature.h"
#include "DatabaseEnv.h"
#include "GameTime.h"
#include "Item.h"
#include "Log.h"
#include "ObjectMgr.h"
#include "Opcodes.h"
#include "Player.h"
#include "ScriptMgr.h"
#include "SharedDefines.h"
#include "WorldPacket.h"
#include "WorldSession.h"
#include "Chat.h"
#include "UpdateFields.h"
#include "QuestDef.h"

#include <cctype>
#include <map>
#include <mutex>
#include <string>
#include <vector>
#include <utility>

namespace TransmogCollection
{
    // -----------------------
    // AddOn
    // -----------------------
    static constexpr char const* TMOG_ADDON_PREFIX = "Transmog";
    static constexpr uint32 TMOG_ADDON_CHUNK_MAX_CHARS = 200;

    static constexpr uint8 TMOG_PRESET_MIN = 1;
    static constexpr uint8 TMOG_PRESET_MAX = 4;

    static uint8 ClampPreset(int32 v)
    {
        if (v < TMOG_PRESET_MIN) return TMOG_PRESET_MIN;
        if (v > TMOG_PRESET_MAX) return TMOG_PRESET_MAX;
        return static_cast<uint8>(v);
    }

    static uint8 GetActivePreset(uint32 guidLow);
    static void SetActivePreset(uint32 guidLow, uint8 preset);
    static void SendAddonMsgToSelf(WorldSession* session, std::string const& payload);
    static void NotifyAddonLearned(Player* player, ItemTemplate const* proto, uint32 itemEntry);

    // -----------------------
    // Config
    // -----------------------
    static bool s_Enabled = true;
    static bool s_UnlockFromGroupRoll = true;
    static bool s_UnlockFromGroupRollPass = true;

    static std::string s_IgnoreAccounts;
    struct AccountRange { uint32 Min = 0; uint32 Max = 0; };
    static std::vector<AccountRange> s_IgnoreAccountRanges;

    // -----------------------
    // Helpery
    // -----------------------
    static std::vector<std::string> Split(std::string const& s, char sep)
    {
        std::vector<std::string> out;
        std::string cur;
        cur.reserve(s.size());
        for (char c : s)
        {
            if (c == sep)
            {
                out.push_back(cur);
                cur.clear();
            }
            else
                cur.push_back(c);
        }
        out.push_back(cur);
        return out;
    }

    static std::string GetItemColorHex(uint32 quality)
    {
        switch (quality)
        {
            case ITEM_QUALITY_POOR:      return "ff9d9d9d";
            case ITEM_QUALITY_NORMAL:    return "ffffffff";
            case ITEM_QUALITY_UNCOMMON:  return "ff1eff00";
            case ITEM_QUALITY_RARE:      return "ff0070dd";
            case ITEM_QUALITY_EPIC:      return "ffa335ee";
            case ITEM_QUALITY_LEGENDARY: return "ffff8000";
            case ITEM_QUALITY_ARTIFACT:  return "ffe6cc80";
            case ITEM_QUALITY_HEIRLOOM:  return "ffe6cc80";
            default:                     return "ffffffff";
        }
    }

    static std::string BuildItemLink(uint32 itemEntry)
    {
        ItemTemplate const* proto = sObjectMgr->GetItemTemplate(itemEntry);
        if (!proto)
            return "[Unknown Item]";

        std::string color = GetItemColorHex(static_cast<uint32>(proto->Quality));
        std::string name  = proto->Name1;

        return "|c" + color +
            "|Hitem:" + std::to_string(itemEntry) + ":0:0:0:0:0:0:0|h[" +
            name + "]|h|r";
    }

    // -----------------------
    // Cross-slot fist výjimka
    // -----------------------
    static bool IsForbiddenCrossFist(std::string const& slotKey, uint32 invType, uint32 subClass)
    {
        // MAINHAND nesmí přijmout OFFHAND-only fist
        if (slotKey == "MAINHAND" && invType == INVTYPE_WEAPONOFFHAND && subClass == ITEM_SUBCLASS_WEAPON_FIST)
            return true;

        // OFFHAND nesmí přijmout MAINHAND-only fist
        if (slotKey == "OFFHAND" && invType == INVTYPE_WEAPONMAINHAND && subClass == ITEM_SUBCLASS_WEAPON_FIST)
            return true;

        return false;
    }

    // -----------------------
    // Account filtr
    // -----------------------
    static std::string Trim(std::string s)
    {
        auto isSpace = [](unsigned char c) { return std::isspace(c) != 0; };
        while (!s.empty() && isSpace(static_cast<unsigned char>(s.front()))) s.erase(s.begin());
        while (!s.empty() && isSpace(static_cast<unsigned char>(s.back())))  s.pop_back();
        return s;
    }

    static void ParseIgnoreAccounts()
    {
        s_IgnoreAccountRanges.clear();

        std::string raw = s_IgnoreAccounts;
        if (raw.empty())
            return;

        auto parts = Split(raw, ',');
        for (std::string p : parts)
        {
            p = Trim(p);
            if (p.empty())
                continue;

            size_t dash = p.find('-');
            try
            {
                if (dash == std::string::npos)
                {
                    uint32 v = static_cast<uint32>(std::stoul(p));
                    s_IgnoreAccountRanges.push_back({ v, v });
                }
                else
                {
                    std::string a = Trim(p.substr(0, dash));
                    std::string b = Trim(p.substr(dash + 1));
                    if (a.empty() || b.empty())
                        continue;

                    uint32 minV = static_cast<uint32>(std::stoul(a));
                    uint32 maxV = static_cast<uint32>(std::stoul(b));
                    if (minV > maxV)
                        std::swap(minV, maxV);

                    s_IgnoreAccountRanges.push_back({ minV, maxV });
                }
            }
            catch (...)
            {
                continue;
            }
        }
    }

    static bool IsIgnoredAccount(uint32 accountId)
    {
        for (auto const& r : s_IgnoreAccountRanges)
            if (accountId >= r.Min && accountId <= r.Max)
                return true;

        return false;
    }

    static void LoadConfig(bool reload)
    {
        s_Enabled = sConfigMgr->GetOption<bool>("TransmogCollection.Enable", true);
        s_UnlockFromGroupRoll = sConfigMgr->GetOption<bool>("TransmogCollection.UnlockFromGroupRoll", true);
        s_UnlockFromGroupRollPass = sConfigMgr->GetOption<bool>("TransmogCollection.UnlockFromGroupRollPass", true);

        s_IgnoreAccounts = sConfigMgr->GetOption<std::string>("TransmogCollection.IgnoreAccounts", "");
        ParseIgnoreAccounts();

        LOG_INFO("module",
            "[TransmogCollection] {}. Enable={} UnlockFromGroupRoll={} UnlockFromGroupRollPass={} IgnoreAccounts='{}'",
            (reload ? "Reloaded" : "Loaded"),
            (s_Enabled ? 1 : 0),
            (s_UnlockFromGroupRoll ? 1 : 0),
            (s_UnlockFromGroupRollPass ? 1 : 0),
            s_IgnoreAccounts);
    }

    // -----------------------
    // Filter
    // -----------------------
    static bool IsExcludedInventoryType(uint32 invType)
    {
        switch (invType)
        {
            case INVTYPE_NECK:
            case INVTYPE_FINGER:
            case INVTYPE_TRINKET:
                return true;
            default:
                return false;
        }
    }

    static bool IsExcludedArmorSubclass(uint32 subClass)
    {
        switch (subClass)
        {
            case ITEM_SUBCLASS_ARMOR_LIBRAM:
            case ITEM_SUBCLASS_ARMOR_IDOL:
            case ITEM_SUBCLASS_ARMOR_TOTEM:
            case ITEM_SUBCLASS_ARMOR_SIGIL:
                return true;
            default:
                return false;
        }
    }

    static bool IsCollectible(ItemTemplate const* proto)
    {
        if (!proto)
            return false;

        if (proto->Class != ITEM_CLASS_ARMOR && proto->Class != ITEM_CLASS_WEAPON)
            return false;

        if (proto->InventoryType == INVTYPE_NON_EQUIP)
            return false;

        if (IsExcludedInventoryType(proto->InventoryType))
            return false;

        if (proto->Class == ITEM_CLASS_ARMOR && IsExcludedArmorSubclass(proto->SubClass))
            return false;

        if (proto->DisplayInfoID == 0)
            return false;

        return true;
    }

    static bool AlreadyLearned(uint32 accountId, uint32 displayId, uint32 inventoryType)
    {
        std::string sql =
            "SELECT 1 FROM customs.transmog_collection "
            "WHERE account_id=" + std::to_string(accountId) +
            " AND display_id=" + std::to_string(displayId) +
            " AND inventory_type=" + std::to_string(inventoryType) +
            " LIMIT 1";

        QueryResult res = WorldDatabase.Query(sql);
        return (res != nullptr);
    }

    static void SendLearnedChat(Player* player, uint32 itemEntry)
    {
        if (!player)
            return;

        WorldSession* session = player->GetSession();
        if (!session)
            return;

        std::string link = BuildItemLink(itemEntry);
        std::string msg = link + " added to transmog collection.";

        WorldPacket data;
        ChatHandler::BuildChatPacket(data, CHAT_MSG_SYSTEM, LANG_UNIVERSAL, player, player, msg);
        session->SendPacket(&data);
    }

    // -----------------------
    // DB insert: collection
    // -----------------------
    static void LearnAppearance(Player* player, uint32 itemEntry, ItemTemplate const* proto, std::string const& /*reasonTag*/)
    {
        if (!s_Enabled || !player || !proto)
            return;

        if (!IsCollectible(proto))
            return;

        WorldSession* session = player->GetSession();
        if (!session)
            return;

        uint32 accountId = session->GetAccountId();
        if (IsIgnoredAccount(accountId))
            return;

        uint32 displayId = proto->DisplayInfoID;

        std::string sql =
            "INSERT IGNORE INTO customs.transmog_collection "
            "(account_id, display_id, item_entry, item_class, item_subclass, inventory_type) VALUES (" +
            std::to_string(accountId) + ", " +
            std::to_string(displayId) + ", " +
            std::to_string(itemEntry) + ", " +
            std::to_string(static_cast<uint32>(proto->Class)) + ", " +
            std::to_string(static_cast<uint32>(proto->SubClass)) + ", " +
            std::to_string(static_cast<uint32>(proto->InventoryType)) +
            ")";

        uint32 invType = static_cast<uint32>(proto->InventoryType);

        if (AlreadyLearned(accountId, displayId, invType))
            return;

        WorldDatabase.Execute(sql);

        SendLearnedChat(player, itemEntry);

        NotifyAddonLearned(player, proto, itemEntry);
    }

    // -----------------------
    // quest collection
    // -----------------------
    static void LearnQuestChoiceRewards(Player* player, Quest const* quest)
    {
        if (!s_Enabled || !player || !quest)
            return;

        uint32 questId = quest->GetQuestId();

        std::string sql =
            "SELECT "
            "RewardChoiceItemID1, RewardChoiceItemID2, RewardChoiceItemID3, "
            "RewardChoiceItemID4, RewardChoiceItemID5, RewardChoiceItemID6, "
            "RewardItem1, RewardItem2, RewardItem3, RewardItem4 "
            "FROM quest_template "
            "WHERE ID=" + std::to_string(questId) + " LIMIT 1";

        QueryResult res = WorldDatabase.Query(sql);
        if (!res)
            return;

        Field* f = res->Fetch();

        for (uint8 i = 0; i < 6; ++i)
        {
            uint32 entry = f[i].Get<uint32>();
            if (!entry)
                continue;

            ItemTemplate const* proto = sObjectMgr->GetItemTemplate(entry);
            if (!proto)
                continue;

            LearnAppearance(player, entry, proto, "quest_choice");
        }

        for (uint8 i = 6; i < 10; ++i)
        {
            uint32 entry = f[i].Get<uint32>();
            if (!entry)
                continue;

            ItemTemplate const* proto = sObjectMgr->GetItemTemplate(entry);
            if (!proto)
                continue;

            LearnAppearance(player, entry, proto, "quest_reward");
        }
    }

    // -----------------------
    // Loot-roll mapping cache
    // -----------------------
    struct RollCacheEntry
    {
        uint32 ItemEntry = 0;
        int64 ExpiresAt = 0;
    };

    static std::mutex s_RollCacheMutex;
    static std::map<ObjectGuid, RollCacheEntry> s_RollCache;

    static int64 NowSec()
    {
        return GameTime::GetGameTime().count();
    }

    static void PruneRollCache(int64 now)
    {
        for (auto it = s_RollCache.begin(); it != s_RollCache.end(); )
        {
            if (it->second.ExpiresAt <= now)
                it = s_RollCache.erase(it);
            else
                ++it;
        }
    }

    static void CacheRollGuid(ObjectGuid itemGuid, uint32 itemEntry)
    {
        if (!itemGuid || itemEntry == 0)
            return;

        int64 now = NowSec();
        int64 ttl = now + 15 * 60;

        std::lock_guard<std::mutex> lock(s_RollCacheMutex);
        PruneRollCache(now);
        s_RollCache[itemGuid] = RollCacheEntry{ itemEntry, ttl };
    }

    static bool TryGetCachedRollEntry(ObjectGuid itemGuid, uint32& outEntry)
    {
        outEntry = 0;
        if (!itemGuid)
            return false;

        int64 now = NowSec();

        std::lock_guard<std::mutex> lock(s_RollCacheMutex);
        PruneRollCache(now);

        auto it = s_RollCache.find(itemGuid);
        if (it == s_RollCache.end())
            return false;

        outEntry = it->second.ItemEntry;
        return (outEntry != 0);
    }

    // -----------------------
    // Packet parsing helpers
    // -----------------------
    static bool IsPlausibleItem(uint32 entry)
    {
        return entry != 0 && sObjectMgr->GetItemTemplate(entry) != nullptr;
    }

    static bool TryParse_CMSG_LOOT_ROLL(WorldPacket const& packet, ObjectGuid& outGuid, uint8& outRollType)
    {
        outGuid.Clear();
        outRollType = 255;

        {
            WorldPacket tmp(packet);
            tmp.rpos(0);

            ObjectGuid guid;
            uint32 slot = 0;
            uint8 rollType = 255;

            if (tmp.size() >= (sizeof(uint64) + sizeof(uint32) + sizeof(uint8)))
            {
                tmp >> guid;
                tmp >> slot;
                tmp >> rollType;

                if (rollType <= 3)
                {
                    outGuid = guid;
                    outRollType = rollType;
                    return true;
                }
            }
        }

        {
            WorldPacket tmp(packet);
            tmp.rpos(0);

            ObjectGuid guid;
            uint8 slot8 = 0;
            uint8 rollType = 255;

            if (tmp.size() >= (sizeof(uint64) + sizeof(uint8) + sizeof(uint8)))
            {
                tmp >> guid;
                tmp >> slot8;
                tmp >> rollType;

                if (rollType <= 3)
                {
                    outGuid = guid;
                    outRollType = rollType;
                    return true;
                }
            }
        }

        return false;
    }

    static bool TryParse_SMSG_LOOT_START_ROLL(WorldPacket const& packet, ObjectGuid& outItemGuid, uint32& outItemEntry)
    {
        outItemGuid.Clear();
        outItemEntry = 0;

        {
            WorldPacket tmp(packet);
            tmp.rpos(0);

            ObjectGuid itemGuid;
            uint32 mapId = 0;
            uint32 slot = 0;
            uint32 itemEntry = 0;
            uint32 randomSuffix = 0;
            int32 randomPropId = 0;
            uint32 countdown = 0;
            uint8 rollMask = 0;

            if (tmp.size() >= (sizeof(uint64) + 4 + 4 + 4 + 4 + 4 + 4 + 1))
            {
                tmp >> itemGuid;
                tmp >> mapId;
                tmp >> slot;
                tmp >> itemEntry;
                tmp >> randomSuffix;
                tmp >> randomPropId;
                tmp >> countdown;
                tmp >> rollMask;

                if (itemGuid && IsPlausibleItem(itemEntry))
                {
                    outItemGuid = itemGuid;
                    outItemEntry = itemEntry;
                    return true;
                }
            }
        }

        {
            WorldPacket tmp(packet);
            tmp.rpos(0);

            uint32 countdown = 0;
            uint32 mapId = 0;
            ObjectGuid itemGuid;
            uint32 slot = 0;
            uint32 itemEntry = 0;
            uint32 randomSuffix = 0;
            int32 randomPropId = 0;
            uint8 rollMask = 0;

            if (tmp.size() >= (4 + 4 + sizeof(uint64) + 4 + 4 + 4 + 4 + 1))
            {
                tmp >> countdown;
                tmp >> mapId;
                tmp >> itemGuid;
                tmp >> slot;
                tmp >> itemEntry;
                tmp >> randomSuffix;
                tmp >> randomPropId;
                tmp >> rollMask;

                if (itemGuid && IsPlausibleItem(itemEntry))
                {
                    outItemGuid = itemGuid;
                    outItemEntry = itemEntry;
                    return true;
                }
            }
        }

        return false;
    }

    // -----------------------
    // Slot mapping
    // -----------------------
    static int32 SlotKeyToEquipSlot(std::string const& key)
    {
        if (key == "HEAD")     return EQUIPMENT_SLOT_HEAD;
        if (key == "SHOULDER") return EQUIPMENT_SLOT_SHOULDERS;
        if (key == "BACK")     return EQUIPMENT_SLOT_BACK;
        if (key == "CHEST")    return EQUIPMENT_SLOT_CHEST;
        if (key == "WRIST")    return EQUIPMENT_SLOT_WRISTS;
        if (key == "HANDS")    return EQUIPMENT_SLOT_HANDS;
        if (key == "WAIST")    return EQUIPMENT_SLOT_WAIST;
        if (key == "LEGS")     return EQUIPMENT_SLOT_LEGS;
        if (key == "FEET")     return EQUIPMENT_SLOT_FEET;
        if (key == "SHIRT")    return EQUIPMENT_SLOT_BODY;
        if (key == "TABARD")   return EQUIPMENT_SLOT_TABARD;
        if (key == "MAINHAND") return EQUIPMENT_SLOT_MAINHAND;
        if (key == "OFFHAND")  return EQUIPMENT_SLOT_OFFHAND;
        if (key == "RANGED")   return EQUIPMENT_SLOT_RANGED;
        return -1;
    }

    static std::string EquipSlotToSlotKey(uint8 slot)
    {
        switch (slot)
        {
            case EQUIPMENT_SLOT_HEAD:      return "HEAD";
            case EQUIPMENT_SLOT_SHOULDERS: return "SHOULDER";
            case EQUIPMENT_SLOT_BACK:      return "BACK";
            case EQUIPMENT_SLOT_CHEST:     return "CHEST";
            case EQUIPMENT_SLOT_WRISTS:    return "WRIST";
            case EQUIPMENT_SLOT_HANDS:     return "HANDS";
            case EQUIPMENT_SLOT_WAIST:     return "WAIST";
            case EQUIPMENT_SLOT_LEGS:      return "LEGS";
            case EQUIPMENT_SLOT_FEET:      return "FEET";
            case EQUIPMENT_SLOT_BODY:      return "SHIRT";
            case EQUIPMENT_SLOT_TABARD:    return "TABARD";
            case EQUIPMENT_SLOT_MAINHAND:  return "MAINHAND";
            case EQUIPMENT_SLOT_OFFHAND:   return "OFFHAND";
            case EQUIPMENT_SLOT_RANGED:    return "RANGED";
            default:                       return "";
        }
    }

    static std::vector<uint32> GetInvTypesForSlotKey(std::string const& key)
    {
        if (key == "HEAD")     return { INVTYPE_HEAD };
        if (key == "SHOULDER") return { INVTYPE_SHOULDERS };
        if (key == "BACK")     return { INVTYPE_CLOAK };
        if (key == "CHEST")    return { INVTYPE_CHEST, INVTYPE_ROBE };
        if (key == "WRIST")    return { INVTYPE_WRISTS };
        if (key == "HANDS")    return { INVTYPE_HANDS };
        if (key == "WAIST")    return { INVTYPE_WAIST };
        if (key == "LEGS")     return { INVTYPE_LEGS };
        if (key == "FEET")     return { INVTYPE_FEET };
        if (key == "SHIRT")    return { INVTYPE_BODY };
        if (key == "TABARD")   return { INVTYPE_TABARD };

        // MAINHAND: chceme i OFFHAND-only (kvůli kolekci), ale fist výjimku řeší SQL/ownership/validate
        if (key == "MAINHAND")
            return { INVTYPE_WEAPON, INVTYPE_WEAPONMAINHAND, INVTYPE_WEAPONOFFHAND, INVTYPE_2HWEAPON };

        // OFFHAND: obecné mapování (learn notify); UI/ownership je dynamické přes GetQueryInvTypesForSlot()
        if (key == "OFFHAND")
            return { INVTYPE_WEAPONOFFHAND, INVTYPE_WEAPONMAINHAND, INVTYPE_SHIELD, INVTYPE_HOLDABLE, INVTYPE_WEAPON, INVTYPE_2HWEAPON };

        if (key == "RANGED")
            return { INVTYPE_RANGED, INVTYPE_RANGEDRIGHT, INVTYPE_THROWN };

        return {};
    }

    // Dynamické invType filtry pro UI/ownership (hlavně OFFHAND)
    static std::vector<uint32> GetQueryInvTypesForSlot(Player* player, std::string const& slotKey)
    {
        if (slotKey != "OFFHAND")
            return GetInvTypesForSlotKey(slotKey);

        if (!player)
            return GetInvTypesForSlotKey(slotKey);

        Item* equipped = player->GetItemByPos(INVENTORY_SLOT_BAG_0, EQUIPMENT_SLOT_OFFHAND);
        if (!equipped || !equipped->GetTemplate())
            return {};

        uint32 eqInv = equipped->GetTemplate()->InventoryType;

        // Shield / Holdable = striktně beze změny
        if (eqInv == INVTYPE_SHIELD || eqInv == INVTYPE_HOLDABLE)
            return { INVTYPE_SHIELD, INVTYPE_HOLDABLE };

        // Jinak (offhand weapon) = zobrazit všechny zbraně (2H + 1H + MH + OH)
        return { INVTYPE_WEAPON, INVTYPE_WEAPONMAINHAND, INVTYPE_WEAPONOFFHAND, INVTYPE_2HWEAPON };
    }

    static void NotifyAddonLearned(Player* player, ItemTemplate const* proto, uint32 itemEntry)
    {
        if (!player || !proto || !s_Enabled)
            return;

        WorldSession* session = player->GetSession();
        if (!session)
            return;

        auto SendLearn = [&](char const* slotKey)
        {
            SendAddonMsgToSelf(session, "LEARN|" + std::string(slotKey) + "|" + std::to_string(itemEntry));
        };

        // Speciální logika pro WEAPONY (kvůli cross-slot + fist výjimkám)
        if (proto->Class == ITEM_CLASS_WEAPON)
        {
            uint32 inv = static_cast<uint32>(proto->InventoryType);
            uint32 sub = static_cast<uint32>(proto->SubClass);

            // Ranged
            if (inv == INVTYPE_RANGED || inv == INVTYPE_RANGEDRIGHT || inv == INVTYPE_THROWN)
            {
                SendLearn("RANGED");
                return;
            }

            // Weapon group
            if (inv == INVTYPE_WEAPONMAINHAND)
            {
                SendLearn("MAINHAND");
                if (!(sub == ITEM_SUBCLASS_WEAPON_FIST)) // OFFHAND nesmí MAINHAND-only fist
                    SendLearn("OFFHAND");
                return;
            }

            if (inv == INVTYPE_WEAPONOFFHAND)
            {
                SendLearn("OFFHAND");
                if (!(sub == ITEM_SUBCLASS_WEAPON_FIST)) // MAINHAND nesmí OFFHAND-only fist
                    SendLearn("MAINHAND");
                return;
            }

            if (inv == INVTYPE_WEAPON || inv == INVTYPE_2HWEAPON)
            {
                // běžné 1H/2H (včetně fist "INVTYPE_WEAPON") chceme v obou kolekcích
                SendLearn("MAINHAND");
                SendLearn("OFFHAND");
                return;
            }
            // cokoliv dalšího necháme spadnout na standard mapping níž
        }

        // Standard mapping (armor + zbytek)
        static char const* kSlotKeys[] =
        {
            "HEAD","SHOULDER","BACK","CHEST","WRIST",
            "HANDS","WAIST","LEGS","FEET","SHIRT","TABARD",
            "MAINHAND","OFFHAND","RANGED"
        };

        uint32 inv = static_cast<uint32>(proto->InventoryType);

        for (char const* slotKey : kSlotKeys)
        {
            auto invTypes = GetInvTypesForSlotKey(slotKey);
            for (uint32 t : invTypes)
            {
                if (t == inv)
                {
                    SendLearn(slotKey);
                    break;
                }
            }
        }
    }

    // -----------------------
    // AddOn send helper (server -> client)
    // -----------------------
    static void SendAddonMsgToSelf(WorldSession* session, std::string const& payload)
    {
        if (!session)
            return;

        Player* plr = session->GetPlayer();
        if (!plr)
            return;

        std::string full = std::string(TMOG_ADDON_PREFIX) + "\t" + payload;

        WorldPacket data;
        ChatHandler::BuildChatPacket(data, CHAT_MSG_WHISPER, LANG_ADDON, plr, plr, full);
        session->SendPacket(&data);
    }

    // -----------------------
    // Query collection for UI
    // -----------------------
    static std::vector<uint32> QueryCollectionForSlot(Player* player, uint32 accountId, std::string const& slotKey)
    {
        std::vector<uint32> out;

        auto invTypes = GetQueryInvTypesForSlot(player, slotKey);
        if (invTypes.empty())
            return out;

        std::string invIn = "(";
        for (size_t i = 0; i < invTypes.size(); ++i)
        {
            if (i) invIn += ",";
            invIn += std::to_string(invTypes[i]);
        }
        invIn += ")";

        std::string forbid = "";
        if (slotKey == "MAINHAND")
        {
            forbid = " AND NOT (inventory_type=" + std::to_string(INVTYPE_WEAPONOFFHAND) +
                     " AND item_subclass=" + std::to_string(ITEM_SUBCLASS_WEAPON_FIST) + ")";
        }
        else if (slotKey == "OFFHAND")
        {
            forbid = " AND NOT (inventory_type=" + std::to_string(INVTYPE_WEAPONMAINHAND) +
                     " AND item_subclass=" + std::to_string(ITEM_SUBCLASS_WEAPON_FIST) + ")";
        }

        std::string sql =
            "SELECT MIN(item_entry) AS item_entry "
            "FROM customs.transmog_collection "
            "WHERE account_id=" + std::to_string(accountId) +
            " AND inventory_type IN " + invIn +
            forbid +
            " GROUP BY display_id "
            "ORDER BY item_entry ASC";

        QueryResult res = WorldDatabase.Query(sql);
        if (!res)
            return out;

        do
        {
            Field* f = res->Fetch();
            uint32 entry = f[0].Get<uint32>();
            if (entry)
                out.push_back(entry);
        } while (res->NextRow());

        return out;
    }

    static void SendCollectionRSP(WorldSession* session, std::string const& slotKey, std::vector<uint32> const& itemEntries)
    {
        if (!session)
            return;

        std::vector<std::string> chunks;
        chunks.reserve(8);

        std::string current;
        current.reserve(TMOG_ADDON_CHUNK_MAX_CHARS + 32);

        for (size_t i = 0; i < itemEntries.size(); ++i)
        {
            std::string add = std::to_string(itemEntries[i]);
            if (!current.empty())
                add = "," + add;

            if (!current.empty() && (current.size() + add.size() > TMOG_ADDON_CHUNK_MAX_CHARS))
            {
                chunks.push_back(current);
                current.clear();

                if (!add.empty() && add[0] == ',')
                    add.erase(0, 1);
            }

            current += add;
        }

        if (!current.empty())
            chunks.push_back(current);

        if (chunks.empty())
            chunks.push_back(std::string(""));

        uint32 total = static_cast<uint32>(chunks.size());
        for (uint32 i = 0; i < total; ++i)
        {
            std::string payload =
                "RSP|" + slotKey + "|" + std::to_string(i + 1) + "/" + std::to_string(total) + "|" + chunks[i];

            SendAddonMsgToSelf(session, payload);
        }
    }

    // -----------------------
    // APPLY logic
    // -----------------------
    static bool AccountOwnsAppearanceForSlot(Player* player, uint32 accountId, std::string const& slotKey, uint32 desiredEntry)
    {
        auto invTypes = GetQueryInvTypesForSlot(player, slotKey);
        if (invTypes.empty())
            return false;

        std::string invIn = "(";
        for (size_t i = 0; i < invTypes.size(); ++i)
        {
            if (i) invIn += ",";
            invIn += std::to_string(invTypes[i]);
        }
        invIn += ")";

        // Vytáhneme invType/subclass, a uděláme stejnou fist výjimku i v ownership checku
        std::string sql =
            "SELECT inventory_type, item_subclass FROM customs.transmog_collection "
            "WHERE account_id=" + std::to_string(accountId) +
            " AND item_entry=" + std::to_string(desiredEntry) +
            " AND inventory_type IN " + invIn +
            " LIMIT 1";

        QueryResult res = WorldDatabase.Query(sql);
        if (!res)
            return false;

        Field* f = res->Fetch();
        uint32 inv = f[0].Get<uint32>();
        uint32 sub = f[1].Get<uint32>();

        if (IsForbiddenCrossFist(slotKey, inv, sub))
            return false;

        return true;
    }

    static void SaveApplied(uint32 guidLow, uint8 preset, uint8 equipSlot, uint32 desiredEntry)
    {
        preset = ClampPreset(preset);

        std::string sql =
            "INSERT INTO customs.transmog_applied (guid, preset, slot, item_entry) VALUES (" +
            std::to_string(guidLow) + ", " +
            std::to_string(static_cast<uint32>(preset)) + ", " +
            std::to_string(static_cast<uint32>(equipSlot)) + ", " +
            std::to_string(desiredEntry) +
            ") ON DUPLICATE KEY UPDATE item_entry=" + std::to_string(desiredEntry);

        WorldDatabase.Execute(sql);
    }

    static void DeleteApplied(uint32 guidLow, uint8 preset, uint8 equipSlot)
    {
        preset = ClampPreset(preset);

        std::string sql =
            "DELETE FROM customs.transmog_applied "
            "WHERE guid=" + std::to_string(guidLow) +
            " AND preset=" + std::to_string(static_cast<uint32>(preset)) +
            " AND slot=" + std::to_string(static_cast<uint32>(equipSlot));

        WorldDatabase.Execute(sql);
    }

    static bool LoadApplied(uint32 guidLow, uint8 preset, uint8 equipSlot, uint32& outEntry)
    {
        preset = ClampPreset(preset);
        outEntry = 0;

        std::string sql =
            "SELECT item_entry FROM customs.transmog_applied "
            "WHERE guid=" + std::to_string(guidLow) +
            " AND preset=" + std::to_string(static_cast<uint32>(preset)) +
            " AND slot=" + std::to_string(static_cast<uint32>(equipSlot)) +
            " LIMIT 1";

        QueryResult res = WorldDatabase.Query(sql);
        if (!res)
            return false;

        Field* f = res->Fetch();
        outEntry = f[0].Get<uint32>();
        return outEntry != 0;
    }

    static void ApplyVisible(Player* player, uint8 equipSlot, uint32 desiredEntry)
    {
        if (!player)
            return;

        Item* equipped = player->GetItemByPos(INVENTORY_SLOT_BAG_0, equipSlot);
        if (!equipped)
            return;

        ItemTemplate const* desiredProto = sObjectMgr->GetItemTemplate(desiredEntry);
        if (!desiredProto)
            return;

        if (!IsCollectible(desiredProto))
            return;

        uint32 enchant = equipped->GetEnchantmentId(PERM_ENCHANTMENT_SLOT);

        uint16 entryField   = static_cast<uint16>(PLAYER_VISIBLE_ITEM_1_ENTRYID + (equipSlot * 2));
        uint16 enchantField = static_cast<uint16>(PLAYER_VISIBLE_ITEM_1_ENCHANTMENT + (equipSlot * 2));

        player->SetUInt32Value(entryField, desiredEntry);
        player->SetUInt32Value(enchantField, enchant);

        player->ForceValuesUpdateAtIndex(entryField);
        player->ForceValuesUpdateAtIndex(enchantField);

        player->ForceValuesUpdateAtIndex(UNIT_FIELD_BYTES_2);
    }

    static void RestoreVisibleFromEquipped(Player* player, uint8 equipSlot)
    {
        if (!player)
            return;

        Item* equipped = player->GetItemByPos(INVENTORY_SLOT_BAG_0, equipSlot);

        uint32 visibleEntry = equipped ? equipped->GetEntry() : 0;
        uint32 enchant = equipped ? equipped->GetEnchantmentId(PERM_ENCHANTMENT_SLOT) : 0;

        uint16 entryField   = static_cast<uint16>(PLAYER_VISIBLE_ITEM_1_ENTRYID + (equipSlot * 2));
        uint16 enchantField = static_cast<uint16>(PLAYER_VISIBLE_ITEM_1_ENCHANTMENT + (equipSlot * 2));

        player->SetUInt32Value(entryField, visibleEntry);
        player->SetUInt32Value(enchantField, enchant);

        player->ForceValuesUpdateAtIndex(entryField);
        player->ForceValuesUpdateAtIndex(enchantField);
        player->ForceValuesUpdateAtIndex(UNIT_FIELD_BYTES_2);
    }

    static void ApplySavedForSlot(Player* player, uint8 equipSlot)
    {
        if (!player)
            return;

        uint32 guidLow = player->GetGUID().GetCounter();
        uint8 preset = GetActivePreset(guidLow);

        uint32 desiredEntry = 0;
        if (!LoadApplied(guidLow, preset, equipSlot, desiredEntry))
            return;

        if (!sObjectMgr->GetItemTemplate(desiredEntry))
            return;

        ApplyVisible(player, equipSlot, desiredEntry);
    }

    static void ApplyAllSaved(Player* player)
    {
        if (!player)
            return;

        static uint8 slots[] =
        {
            EQUIPMENT_SLOT_HEAD,
            EQUIPMENT_SLOT_SHOULDERS,
            EQUIPMENT_SLOT_BACK,
            EQUIPMENT_SLOT_CHEST,
            EQUIPMENT_SLOT_WRISTS,
            EQUIPMENT_SLOT_HANDS,
            EQUIPMENT_SLOT_WAIST,
            EQUIPMENT_SLOT_LEGS,
            EQUIPMENT_SLOT_FEET,
            EQUIPMENT_SLOT_BODY,
            EQUIPMENT_SLOT_TABARD,
            EQUIPMENT_SLOT_MAINHAND,
            EQUIPMENT_SLOT_OFFHAND,
            EQUIPMENT_SLOT_RANGED
        };

        for (uint8 s : slots)
            ApplySavedForSlot(player, s);
    }

    static void ClearAllApplied(Player* player)
    {
        if (!player)
            return;

        uint32 guidLow = player->GetGUID().GetCounter();

        WorldDatabase.Execute(
            "DELETE FROM customs.transmog_applied WHERE guid=" + std::to_string(guidLow));

        WorldDatabase.Execute(
            "DELETE FROM customs.transmog_preset_state WHERE guid=" + std::to_string(guidLow));

        static uint8 slots[] =
        {
            EQUIPMENT_SLOT_HEAD, EQUIPMENT_SLOT_SHOULDERS, EQUIPMENT_SLOT_BACK, EQUIPMENT_SLOT_CHEST,
            EQUIPMENT_SLOT_WRISTS, EQUIPMENT_SLOT_HANDS, EQUIPMENT_SLOT_WAIST, EQUIPMENT_SLOT_LEGS,
            EQUIPMENT_SLOT_FEET, EQUIPMENT_SLOT_BODY, EQUIPMENT_SLOT_TABARD,
            EQUIPMENT_SLOT_MAINHAND, EQUIPMENT_SLOT_OFFHAND, EQUIPMENT_SLOT_RANGED
        };

        for (uint8 s : slots)
            RestoreVisibleFromEquipped(player, s);
    }

    // -----------------------
    // Weapon strict check (MAINHAND/OFFHAND/RANGED)
    // -----------------------
    static bool ValidateWeaponStrict(Player* player, uint8 equipSlot, uint32 desiredEntry, std::string& outReason)
    {
        outReason.clear();

        Item* equipped = player->GetItemByPos(INVENTORY_SLOT_BAG_0, equipSlot);
        if (!equipped)
        {
            outReason = "NO_ITEM_EQUIPPED";
            return false;
        }

        ItemTemplate const* eq = equipped->GetTemplate();
        ItemTemplate const* des = sObjectMgr->GetItemTemplate(desiredEntry);
        if (!eq || !des)
        {
            outReason = "BAD_TEMPLATE";
            return false;
        }

        if (equipSlot == EQUIPMENT_SLOT_MAINHAND)
        {
            uint32 desInv = des->InventoryType;

            // MAINHAND povoluje i OFFHAND-only, ale NE pokud je to fist
            if (desInv == INVTYPE_WEAPONOFFHAND && des->SubClass == ITEM_SUBCLASS_WEAPON_FIST)
            {
                outReason = "MAINHAND_OFFHAND_FIST_FORBIDDEN";
                return false;
            }

            if (desInv == INVTYPE_WEAPON ||
                desInv == INVTYPE_WEAPONMAINHAND ||
                desInv == INVTYPE_WEAPONOFFHAND ||
                desInv == INVTYPE_2HWEAPON)
                return true;

            outReason = "MAINHAND_WEAPON_ONLY";
            return false;
        }

        if (equipSlot == EQUIPMENT_SLOT_OFFHAND)
        {
            uint32 eqInv  = eq->InventoryType;
            uint32 desInv = des->InventoryType;

            // Shield / Holdable zůstává striktně
            if (eqInv == INVTYPE_SHIELD || eqInv == INVTYPE_HOLDABLE)
            {
                if (desInv != INVTYPE_SHIELD && desInv != INVTYPE_HOLDABLE)
                {
                    outReason = "OFFHAND_SHIELD_OR_HOLDABLE_ONLY";
                    return false;
                }
                return true;
            }

            // OFFHAND weapon: povolit všechny zbraně, ale NE pokud je to MAINHAND-only fist
            if (desInv == INVTYPE_WEAPONMAINHAND && des->SubClass == ITEM_SUBCLASS_WEAPON_FIST)
            {
                outReason = "OFFHAND_MAINHAND_FIST_FORBIDDEN";
                return false;
            }

            if (desInv == INVTYPE_2HWEAPON ||
                desInv == INVTYPE_WEAPON ||
                desInv == INVTYPE_WEAPONMAINHAND ||
                desInv == INVTYPE_WEAPONOFFHAND)
                return true;

            outReason = "OFFHAND_WEAPON_ONLY";
            return false;
        }

        if (equipSlot == EQUIPMENT_SLOT_RANGED)
        {
            auto isRangedInv = [](uint32 inv) -> bool
            {
                return inv == INVTYPE_RANGED || inv == INVTYPE_RANGEDRIGHT || inv == INVTYPE_THROWN;
            };

            if (!isRangedInv(eq->InventoryType) || !isRangedInv(des->InventoryType))
            {
                outReason = "RANGED_INVTYPE_MISMATCH";
                return false;
            }

            auto isRangedMixGroup = [](uint32 sub) -> bool
            {
                return sub == ITEM_SUBCLASS_WEAPON_BOW
                    || sub == ITEM_SUBCLASS_WEAPON_GUN
                    || sub == ITEM_SUBCLASS_WEAPON_CROSSBOW
                    || sub == ITEM_SUBCLASS_WEAPON_WAND
                    || sub == ITEM_SUBCLASS_WEAPON_THROWN;
            };

            if (isRangedMixGroup(eq->SubClass) && isRangedMixGroup(des->SubClass))
                return true;

            if (eq->SubClass != des->SubClass)
            {
                outReason = "RANGED_SUBCLASS_MISMATCH";
                return false;
            }

            return true;
        }

        return true;
    }

    // -----------------------
    // Send applied state to AddOn
    // -----------------------
    static void SendAppliedToAddon(WorldSession* session)
    {
        if (!session)
            return;

        Player* player = session->GetPlayer();
        if (!player)
            return;

        uint32 guidLow = player->GetGUID().GetCounter();
        uint8 preset = GetActivePreset(guidLow);

        std::string sql =
            "SELECT slot, item_entry FROM customs.transmog_applied "
            "WHERE guid=" + std::to_string(guidLow) +
            " AND preset=" + std::to_string(static_cast<uint32>(preset));

        QueryResult res = WorldDatabase.Query(sql);
        if (res)
        {
            do
            {
                Field* f = res->Fetch();
                uint8 slot = f[0].Get<uint8>();
                uint32 entry = f[1].Get<uint32>();

                std::string slotKey = EquipSlotToSlotKey(slot);
                if (!slotKey.empty() && entry > 0)
                    SendAddonMsgToSelf(session, "APPLIED|" + slotKey + "|" + std::to_string(entry));
            } while (res->NextRow());
        }

        SendAddonMsgToSelf(session, "APPLIED_DONE");
    }

    // -----------------------
    // AddOn receive parser
    // -----------------------
    static bool TryExtractPayloadFromString(std::string const& s, std::string& outPayload)
    {
        size_t tab = s.find('\t');
        if (tab == std::string::npos)
            return false;

        std::string prefix = s.substr(0, tab);
        if (prefix != TMOG_ADDON_PREFIX)
            return false;

        outPayload = s.substr(tab + 1);
        return true;
    }

    static bool TryParse_AddonFrom_CMSG_MESSAGECHAT(WorldPacket const& packet, std::string& outMsg)
    {
        outMsg.clear();

        WorldPacket tmp(packet);
        tmp.rpos(0);

        uint32 chatType = 0;
        int32 lang = 0;

        if (tmp.size() < 8)
            return false;

        tmp >> chatType;
        tmp >> lang;

        if (lang != LANG_ADDON)
            return false;

        if (chatType == CHAT_MSG_CHANNEL)
        {
            if (tmp.rpos() >= tmp.size())
                return false;

            std::string channelName;
            tmp >> channelName;
        }
        else if (chatType == CHAT_MSG_WHISPER)
        {
            if (tmp.rpos() >= tmp.size())
                return false;

            std::string targetName;
            tmp >> targetName;
        }

        std::vector<std::string> parts;
        parts.reserve(4);

        while (tmp.rpos() < tmp.size() && parts.size() < 4)
        {
            std::string s;
            tmp >> s;
            if (!s.empty())
                parts.push_back(s);
        }

        for (auto const& s : parts)
        {
            std::string payload;
            if (TryExtractPayloadFromString(s, payload))
            {
                outMsg = payload;
                return true;
            }
        }

        for (size_t i = 0; i + 1 < parts.size(); ++i)
        {
            if (parts[i] == TMOG_ADDON_PREFIX)
            {
                outMsg = parts[i + 1];
                return true;
            }
        }

        return false;
    }

    static uint8 GetActivePreset(uint32 guidLow)
    {
        std::string sql =
            "SELECT active_preset FROM customs.transmog_preset_state "
            "WHERE guid=" + std::to_string(guidLow) + " LIMIT 1";

        QueryResult res = WorldDatabase.Query(sql);
        if (!res)
            return 1;

        Field* f = res->Fetch();
        return ClampPreset(f[0].Get<uint8>());
    }

    static void SetActivePreset(uint32 guidLow, uint8 preset)
    {
        preset = ClampPreset(preset);

        std::string sql =
            "INSERT INTO customs.transmog_preset_state (guid, active_preset) VALUES (" +
            std::to_string(guidLow) + ", " + std::to_string(static_cast<uint32>(preset)) +
            ") ON DUPLICATE KEY UPDATE active_preset=" + std::to_string(static_cast<uint32>(preset));

        WorldDatabase.Execute(sql);
    }

    // -----------------------
    // Scripts
    // -----------------------
    class TC_WorldScript final : public WorldScript
    {
    public:
        TC_WorldScript() : WorldScript("TransmogCollection_WorldScript") { }

        void OnBeforeConfigLoad(bool reload) override
        {
            LoadConfig(reload);
        }
    };

    class TC_PlayerScript final : public PlayerScript
    {
    public:
        TC_PlayerScript() : PlayerScript("TransmogCollection_PlayerScript") { }

        void OnPlayerLogin(Player* player) override
        {
            if (!s_Enabled || !player)
                return;

            ApplyAllSaved(player);
        }

        void OnPlayerStoreNewItem(Player* player, Item* item, uint32 /*count*/) override
        {
            if (!s_Enabled || !player || !item)
                return;

            ItemTemplate const* proto = item->GetTemplate();
            if (!proto)
                return;

            LearnAppearance(player, item->GetEntry(), proto, "store_new_item");
        }

        void OnPlayerAfterStoreOrEquipNewItem(Player* player, uint32 itemId, Item* item, uint8 /*bag*/, uint8 /*slot*/, uint8 /*mode*/,
            ItemTemplate const* itemTemplate, Creature* /*creature*/, VendorItem const* /*vendorItem*/, bool /*update*/) override
        {
            if (!s_Enabled || !player)
                return;

            ItemTemplate const* proto = itemTemplate;
            if (!proto && item)
                proto = item->GetTemplate();

            LearnAppearance(player, itemId, proto, "store_or_equip");

            ApplyAllSaved(player);
        }

        void OnPlayerEquip(Player* player, Item* item, uint8 /*bag*/, uint8 /*slot*/, bool /*update*/) override
        {
            if (!s_Enabled || !player || !item)
                return;

            ItemTemplate const* proto = item->GetTemplate();
            if (!proto)
                return;

            LearnAppearance(player, item->GetEntry(), proto, "equip");
        }

        void OnPlayerCompleteQuest(Player* player, Quest const* quest) override
        {
            if (!s_Enabled || !player || !quest)
                return;

            LearnQuestChoiceRewards(player, quest);
        }
    };

    class TC_ServerScript final : public ServerScript
    {
    public:
        TC_ServerScript() : ServerScript("TransmogCollection_ServerScript") { }

        bool CanPacketSend(WorldSession* session, WorldPacket& packet) override
        {
            if (!s_Enabled || !session)
                return true;

            if (s_UnlockFromGroupRoll && packet.GetOpcode() == SMSG_LOOT_START_ROLL)
            {
                ObjectGuid itemGuid;
                uint32 itemEntry = 0;

                if (TryParse_SMSG_LOOT_START_ROLL(packet, itemGuid, itemEntry))
                    CacheRollGuid(itemGuid, itemEntry);
            }

            return true;
        }

        bool CanPacketReceive(WorldSession* session, WorldPacket& packet) override
        {
            if (!s_Enabled || !session)
                return true;

            // 1) Loot roll unlock
            if (s_UnlockFromGroupRoll && packet.GetOpcode() == CMSG_LOOT_ROLL)
            {
                ObjectGuid itemGuid;
                uint8 rollType = 255;

                if (!TryParse_CMSG_LOOT_ROLL(packet, itemGuid, rollType))
                    return true;

                if (rollType == 0 && !s_UnlockFromGroupRollPass)
                    return true;

                uint32 itemEntry = 0;
                if (!TryGetCachedRollEntry(itemGuid, itemEntry))
                    return true;

                ItemTemplate const* proto = sObjectMgr->GetItemTemplate(itemEntry);
                if (!proto)
                    return true;

                Player* player = session->GetPlayer();
                if (!player)
                    return true;

                LearnAppearance(player, itemEntry, proto, "loot_roll");
                return true;
            }

            // 2) AddOn endpoint
            if (packet.GetOpcode() == CMSG_MESSAGECHAT)
            {
                std::string msg;
                if (!TryParse_AddonFrom_CMSG_MESSAGECHAT(packet, msg))
                    return true;

                Player* player = session->GetPlayer();
                if (!player)
                    return true;

                if (msg == "GET_APPLIED")
                {
                    SendAppliedToAddon(session);
                    return true;
                }

                if (msg.rfind("SET_PRESET|", 0) == 0)
                {
                    uint32 guidLow = player->GetGUID().GetCounter();

                    uint8 preset = 1;
                    try
                    {
                        preset = ClampPreset(static_cast<int32>(std::stoul(msg.substr(11))));
                    }
                    catch (...)
                    {
                        preset = 1;
                    }

                    SetActivePreset(guidLow, preset);

                    ApplyAllSaved(player);

                    SendAddonMsgToSelf(session, "PRESET_OK|" + std::to_string(static_cast<uint32>(preset)));
                    return true;
                }

                if (msg.rfind("CLEAR|", 0) == 0)
                {
                    auto parts = Split(msg, '|');
                    if (parts.size() < 2)
                        return true;

                    std::string slotKey = parts[1];
                    int32 equipSlot = SlotKeyToEquipSlot(slotKey);
                    if (equipSlot < 0)
                        return true;

                    uint32 guidLow = player->GetGUID().GetCounter();

                    uint8 preset = GetActivePreset(guidLow);
                    if (parts.size() >= 3)
                    {
                        try { preset = ClampPreset(static_cast<int32>(std::stoul(parts[2]))); }
                        catch (...) { preset = GetActivePreset(guidLow); }
                        SetActivePreset(guidLow, preset);
                    }

                    DeleteApplied(guidLow, preset, static_cast<uint8>(equipSlot));
                    RestoreVisibleFromEquipped(player, static_cast<uint8>(equipSlot));

                    SendAddonMsgToSelf(session, "APPLIED|" + slotKey + "|0");
                    SendAddonMsgToSelf(session, "CLEAR_OK|" + slotKey);
                    return true;
                }

                if (msg.rfind("REQ|", 0) == 0 && msg.size() > 4)
                {
                    std::string slotKey = msg.substr(4);
                    uint32 accountId = session->GetAccountId();

                    auto list = QueryCollectionForSlot(player, accountId, slotKey);
                    SendCollectionRSP(session, slotKey, list);
                    return true;
                }

                if (msg.rfind("APPLY|", 0) == 0)
                {
                    auto parts = Split(msg, '|');
                    if (parts.size() != 3 && parts.size() != 4)
                        return true;

                    std::string slotKey = parts[1];

                    uint32 desiredEntry = 0;
                    try
                    {
                        desiredEntry = static_cast<uint32>(std::stoul(parts[2]));
                    }
                    catch (...)
                    {
                        return true;
                    }

                    int32 equipSlot = SlotKeyToEquipSlot(slotKey);
                    if (equipSlot < 0)
                        return true;

                    if (!IsPlausibleItem(desiredEntry))
                        return true;

                    uint32 accountId = session->GetAccountId();
                    if (!AccountOwnsAppearanceForSlot(player, accountId, slotKey, desiredEntry))
                        return true;

                    if (equipSlot == EQUIPMENT_SLOT_MAINHAND || equipSlot == EQUIPMENT_SLOT_OFFHAND || equipSlot == EQUIPMENT_SLOT_RANGED)
                    {
                        std::string reason;
                        if (!ValidateWeaponStrict(player, static_cast<uint8>(equipSlot), desiredEntry, reason))
                        {
                            SendAddonMsgToSelf(session, "APPLY_ERR|" + reason);
                            return true;
                        }
                    }

                    uint32 guidLow = player->GetGUID().GetCounter();

                    uint8 preset = GetActivePreset(guidLow);
                    if (parts.size() == 4)
                    {
                        try { preset = ClampPreset(static_cast<int32>(std::stoul(parts[3]))); }
                        catch (...) { preset = GetActivePreset(guidLow); }

                        SetActivePreset(guidLow, preset);
                    }

                    SaveApplied(guidLow, preset, static_cast<uint8>(equipSlot), desiredEntry);
                    ApplyVisible(player, static_cast<uint8>(equipSlot), desiredEntry);

                    SendAddonMsgToSelf(session, "APPLY_OK|" + slotKey + "|" + std::to_string(desiredEntry));
                    return true;
                }

                return true;
            }

            return true;
        }
    };
}

// -----------------------
// Entry
// -----------------------
void Addmod_transmog_collectionScripts()
{
    new TransmogCollection::TC_WorldScript();
    new TransmogCollection::TC_PlayerScript();
    new TransmogCollection::TC_ServerScript();
}
